<?php
/**
 * Products tabs map.
 *
 * @package xts
 */

namespace XTS\Elementor;

use Elementor\Controls_Manager;
use Elementor\Group_Control_Image_Size;
use Elementor\Group_Control_Typography;
use Elementor\Group_Control_Box_Shadow;
use Elementor\Group_Control_Border;
use Elementor\Plugin;
use Elementor\Repeater;
use Elementor\Widget_Base;

if ( ! defined( 'ABSPATH' ) ) {
	exit; // Direct access not allowed.
}

/**
 * Elementor widget that inserts an embeddable content into the page, from any given URL.
 *
 * @since 1.0.0
 */
class Products_Tabs extends Widget_Base {
	/**
	 * Get widget name.
	 *
	 * @since 1.0.0
	 * @access public
	 *
	 * @return string Widget name.
	 */
	public function get_name() {
		return 'wd_products_tabs';
	}

	/**
	 * Get widget title.
	 *
	 * @since 1.0.0
	 * @access public
	 *
	 * @return string Widget title.
	 */
	public function get_title() {
		return esc_html__( 'AJAX Products tabs', 'studiare-plus' );
	}

	/**
	 * Get widget icon.
	 *
	 * @since 1.0.0
	 * @access public
	 *
	 * @return string Widget icon.
	 */
	public function get_icon() {
		return 'wd-icon-product-tabs';
	}

	/**
	 * Get widget categories.
	 *
	 * @since 1.0.0
	 * @access public
	 *
	 * @return array Widget categories.
	 */
	public function get_categories() {
		return [ 'wd-elements' ];
	}

	/**
	 * Get attribute taxonomies
	 *
	 * @since 1.0.0
	 */
	public function get_product_attributes_array() {
		$attributes = [];

		if ( taxonomy_exists( 'product_brand' ) ) {
			$attributes[] = 'product_brand';
		}

		if ( woodmart_woocommerce_installed() ) {
			foreach ( wc_get_attribute_taxonomies() as $attribute ) {
				$attributes[] = 'pa_' . $attribute->attribute_name;
			}
		}

		return $attributes;
	}

	/**
	 * Register the widget controls.
	 *
	 * @since 1.0.0
	 * @access protected
	 */
	protected function register_controls() {
		/**
		 * Content tab.
		 */

		/**
		 * General settings.
		 */
		$this->start_controls_section(
			'general_content_section',
			[
				'label' => esc_html__( 'Heading', 'studiare-plus' ),
			]
		);

		$this->add_control(
			'title',
			[
				'label'   => esc_html__( 'Title', 'studiare-plus' ),
				'type'    => Controls_Manager::TEXT,
				'default' => 'Title text example',
			]
		);

		$this->add_control(
			'description',
			[
				'label'     => esc_html__( 'Description', 'studiare-plus' ),
				'type'      => Controls_Manager::TEXTAREA,
				'condition' => [
					'design' => array( 'default', 'aside' ),
				],
			]
		);

		$this->add_control(
			'title_icon_options_separator',
			array(
				'label'     => esc_html__( 'Title icon', 'studiare-plus' ),
				'type'      => Controls_Manager::HEADING,
				'separator' => 'before',
			)
		);

		$this->add_control(
			'image',
			[
				'label' => esc_html__( 'Choose image', 'studiare-plus' ),
				'type'  => Controls_Manager::MEDIA,
			]
		);

		$this->add_group_control(
			Group_Control_Image_Size::get_type(),
			[
				'name'      => 'image',
				'default'   => 'thumbnail',
				'separator' => 'none',
			]
		);

		$this->end_controls_section();

		/**
		 * Tabs settings.
		 */
		$this->start_controls_section(
			'tabs_content_section',
			[
				'label' => esc_html__( 'Tabs', 'studiare-plus' ),
			]
		);

		$repeater = new Repeater();

		$repeater->start_controls_tabs( 'content_tabs' );

		$repeater->start_controls_tab(
			'query_tab',
			[
				'label' => esc_html__( 'Query', 'studiare-plus' ),
			]
		);

		$repeater->add_control(
			'post_type',
			[
				'label'       => esc_html__( 'Data source', 'studiare-plus' ),
				'description' => esc_html__( 'Select content type for your grid.', 'studiare-plus' ),
				'type'        => Controls_Manager::SELECT,
				'default'     => 'product',
				'options'     => array(
					'product'            => esc_html__( 'All Products', 'studiare-plus' ),
					'featured'           => esc_html__( 'Featured Products', 'studiare-plus' ),
					'sale'               => esc_html__( 'Sale Products', 'studiare-plus' ),
					'new'                => esc_html__( 'Products with NEW label', 'studiare-plus' ),
					'bestselling'        => esc_html__( 'Bestsellers', 'studiare-plus' ),
					'ids'                => esc_html__( 'List of IDs', 'studiare-plus' ),
					'top_rated_products' => esc_html__( 'Top Rated Products', 'studiare-plus' ),
				),
			]
		);

		$repeater->add_control(
			'ajax_recently_viewed',
			[
				'label'        => esc_html__( 'Update with AJAX on page load', 'studiare-plus' ),
				'description'  => esc_html__( 'Enable this option if you use full-page cache like WP Rocket.', 'studiare-plus' ),
				'type'         => Controls_Manager::SWITCHER,
				'default'      => 'no',
				'label_on'     => esc_html__( 'Yes', 'studiare-plus' ),
				'label_off'    => esc_html__( 'No', 'studiare-plus' ),
				'return_value' => 'yes',
				'condition'    => [
					'post_type' => 'recently_viewed',
				],
			]
		);

		$repeater->add_control(
			'include',
			[
				'label'       => esc_html__( 'Include only', 'studiare-plus' ),
				'description' => esc_html__( 'Add products by title.', 'studiare-plus' ),
				'type'        => 'wd_autocomplete',
				'search'      => 'woodmart_get_posts_by_query',
				'render'      => 'woodmart_get_posts_title_by_id',
				'post_type'   => 'product',
				'multiple'    => true,
				'label_block' => true,
				'condition'   => [
					'post_type' => 'ids',
				],
			]
		);

		$repeater->add_control(
			'taxonomies',
			[
				'label'       => esc_html__( 'Categories or tags', 'studiare-plus' ),
				'description' => esc_html__( 'List of product categories.', 'studiare-plus' ),
				'type'        => 'wd_autocomplete',
				'search'      => 'woodmart_get_taxonomies_by_query',
				'render'      => 'woodmart_get_taxonomies_title_by_id',
				'taxonomy'    => array_merge( [ 'product_cat', 'product_tag' ], $this->get_product_attributes_array() ),
				'multiple'    => true,
				'label_block' => true,
				'condition'   => [
					'post_type!' => 'ids',
				],
			]
		);

		$repeater->add_control(
			'orderby',
			[
				'label'       => esc_html__( 'Order by', 'studiare-plus' ),
				'description' => esc_html__( 'Select order type. If "Meta value" or "Meta value Number" is chosen then meta key is required.', 'studiare-plus' ),
				'type'        => Controls_Manager::SELECT,
				'default'     => '',
				'options'     => array(
					''               => '',
					'date'           => esc_html__( 'Date', 'studiare-plus' ),
					'id'             => esc_html__( 'ID', 'studiare-plus' ),
					'author'         => esc_html__( 'Author', 'studiare-plus' ),
					'title'          => esc_html__( 'Title', 'studiare-plus' ),
					'modified'       => esc_html__( 'Last modified date', 'studiare-plus' ),
					'comment_count'  => esc_html__( 'Number of comments', 'studiare-plus' ),
					'menu_order'     => esc_html__( 'Menu order', 'studiare-plus' ),
					'meta_value'     => esc_html__( 'Meta value', 'studiare-plus' ),
					'meta_value_num' => esc_html__( 'Meta value number', 'studiare-plus' ),
					'rand'           => esc_html__( 'Random order', 'studiare-plus' ),
					'price'          => esc_html__( 'Price', 'studiare-plus' ),
				),
				'condition'   => [
					'post_type!' => 'recently_viewed',
				],
			]
		);

		$repeater->add_control(
			'offset',
			[
				'label'       => esc_html__( 'Offset', 'studiare-plus' ),
				'description' => esc_html__( 'Number of grid elements to displace or pass over.', 'studiare-plus' ),
				'type'        => Controls_Manager::TEXT,
				'condition'   => [
					'post_type!' => array( 'ids', 'recently_viewed' ),
				],
			]
		);

		$repeater->add_control(
			'query_type',
			[
				'label'     => esc_html__( 'Query type', 'studiare-plus' ),
				'type'      => Controls_Manager::SELECT,
				'default'   => 'OR',
				'options'   => array(
					'OR'  => esc_html__( 'OR', 'studiare-plus' ),
					'AND' => esc_html__( 'AND', 'studiare-plus' ),
				),
				'condition' => array(
					'post_type!' => 'recently_viewed',
				),
			]
		);

		$repeater->add_control(
			'order',
			[
				'label'       => esc_html__( 'Sort order', 'studiare-plus' ),
				'description' => 'Designates the ascending or descending order. More at <a href="http://codex.wordpress.org/Class_Reference/WP_Query#Order_.26_Orderby_Parameters" target="_blank">WordPress codex page</a>.',
				'type'        => Controls_Manager::SELECT,
				'default'     => '',
				'options'     => array(
					''     => esc_html__( 'Inherit', 'studiare-plus' ),
					'DESC' => esc_html__( 'Descending', 'studiare-plus' ),
					'ASC'  => esc_html__( 'Ascending', 'studiare-plus' ),
				),
				'condition'   => [
					'post_type!' => array( 'ids', 'recently_viewed' ),
				],
			]
		);

		$repeater->add_control(
			'hide_out_of_stock',
			[
				'label'        => esc_html__( 'Hide out of stock products', 'studiare-plus' ),
				'type'         => Controls_Manager::SWITCHER,
				'default'      => 'no',
				'label_on'     => esc_html__( 'Yes', 'studiare-plus' ),
				'label_off'    => esc_html__( 'No', 'studiare-plus' ),
				'return_value' => 'yes',
			]
		);

		$repeater->add_control(
			'meta_key',
			[
				'label'       => esc_html__( 'Meta key', 'studiare-plus' ),
				'description' => esc_html__( 'Input meta key for grid ordering.', 'studiare-plus' ),
				'type'        => Controls_Manager::TEXTAREA,
				'condition'   => [
					'orderby' => [ 'meta_value', 'meta_value_num' ],
				],
			]
		);

		$repeater->add_control(
			'exclude',
			[
				'label'       => esc_html__( 'Exclude', 'studiare-plus' ),
				'description' => esc_html__( 'Exclude posts, pages, etc. by title.', 'studiare-plus' ),
				'type'        => 'wd_autocomplete',
				'search'      => 'woodmart_get_posts_by_query',
				'render'      => 'woodmart_get_posts_title_by_id',
				'post_type'   => 'product',
				'multiple'    => true,
				'label_block' => true,
				'condition'   => [
					'post_type!' => 'ids',
				],
			]
		);

		$repeater->end_controls_tab();

		$repeater->start_controls_tab(
			'text_tab',
			[
				'label' => esc_html__( 'Text', 'studiare-plus' ),
			]
		);

		$repeater->add_control(
			'title',
			[
				'label'   => esc_html__( 'Tabs title', 'studiare-plus' ),
				'type'    => Controls_Manager::TEXT,
				'default' => 'Tab title',
			]
		);

		$repeater->end_controls_tab();

		$repeater->start_controls_tab(
			'icon_tab',
			[
				'label' => esc_html__( 'Icon', 'studiare-plus' ),
			]
		);

		$repeater->add_control(
			'icon_type',
			array(
				'label'   => esc_html__( 'Icon type', 'studiare-plus' ),
				'type'    => Controls_Manager::SELECT,
				'options' => array(
					'icon'  => esc_html__( 'With icon', 'studiare-plus' ),
					'image' => esc_html__( 'With image', 'studiare-plus' ),
				),
				'default' => 'image',
			)
		);

		$repeater->add_control(
			'icon',
			array(
				'label'     => esc_html__( 'Icon', 'studiare-plus' ),
				'type'      => Controls_Manager::ICONS,
				'condition' => array(
					'icon_type' => array( 'icon' ),
				),
			)
		);

		$repeater->add_control(
			'image',
			[
				'label'     => esc_html__( 'Choose image', 'studiare-plus' ),
				'type'      => Controls_Manager::MEDIA,
				'condition' => array(
					'icon_type' => array( 'image' ),
				),
			]
		);

		$repeater->add_group_control(
			Group_Control_Image_Size::get_type(),
			[
				'name'      => 'image',
				'default'   => 'thumbnail',
				'separator' => 'none',
				'condition' => array(
					'icon_type' => array( 'image' ),
				),
			]
		);

		$repeater->end_controls_tab();

		$repeater->end_controls_tabs();

		$this->add_control(
			'tabs_items',
			[
				'type'        => Controls_Manager::REPEATER,
				'title_field' => '{{{ title }}}',
				'fields'      => $repeater->get_controls(),
				'default'     => [
					[
						'title' => 'Tab title 1',
					],
					[
						'title' => 'Tab title 2',
					],
					[
						'title' => 'Tab title 3',
					],
				],
			]
		);

		$this->end_controls_section();

		/**
		 * Style tab.
		 */
		/**
		 * Heading settings.
		 */
		$this->start_controls_section(
			'heading_style_section',
			[
				'label' => esc_html__( 'Heading', 'studiare-plus' ),
				'tab'   => Controls_Manager::TAB_STYLE,
			]
		);

		$this->add_control(
			'design',
			[
				'label'   => esc_html__( 'Design', 'studiare-plus' ),
				'type'    => Controls_Manager::SELECT,
				'options' => [
					'default' => esc_html__( 'Default', 'studiare-plus' ),
					'simple'  => esc_html__( 'Bordered', 'studiare-plus' ),
					'alt'     => esc_html__( 'Space between', 'studiare-plus' ),
					'aside'   => esc_html__( 'Aside', 'studiare-plus' ),
				],
				'default' => 'default',
			]
		);

		$this->add_control(
			'tabs_side_width',
			array(
				'label'      => esc_html__( 'Side width', 'studiare-plus' ),
				'type'       => Controls_Manager::SLIDER,
				'size_units' => array( 'px', '%' ),
				'range'      => array(
					'px' => array(
						'min'  => 100,
						'max'  => 500,
						'step' => 1,
					),
					'%'  => array(
						'min'  => 1,
						'max'  => 100,
						'step' => 1,
					),
				),
				'selectors'  => array(
					'{{WRAPPER}} .wd-tabs.tabs-design-aside' => '--wd-side-width: {{SIZE}}{{UNIT}};',
				),
				'condition'  => array(
					'design' => array( 'aside' ),
				),
			)
		);

		$this->add_responsive_control(
			'space_between_tabs_title_vertical',
			array(
				'label'     => esc_html__( 'Spacing', 'studiare-plus' ),
				'type'      => Controls_Manager::SLIDER,
				'range'     => array(
					'px' => array(
						'min'  => 0,
						'max'  => 150,
						'step' => 1,
					),
				),
				'selectors' => array(
					'{{WRAPPER}} .wd-tabs' => '--wd-row-gap: {{SIZE}}px;',
				),
				'devices'   => array( 'desktop', 'tablet', 'mobile' ),
				'classes'   => 'wd-hide-custom-breakpoints',
			)
		);

		$this->add_control(
			'enable_heading_bg',
			array(
				'label'        => esc_html__( 'Background color', 'studiare-plus' ),
				'type'         => Controls_Manager::SWITCHER,
				'default'      => 'no',
				'label_on'     => esc_html__( 'Yes', 'studiare-plus' ),
				'label_off'    => esc_html__( 'No', 'studiare-plus' ),
				'return_value' => 'yes',
			)
		);

		$this->add_control(
			'heading_bg',
			array(
				'label'     => esc_html__( 'Background color', 'studiare-plus' ),
				'type'      => Controls_Manager::COLOR,
				'selectors' => array(
					'{{WRAPPER}} .wd-tabs.wd-header-with-bg .wd-tabs-header' => 'background-color: {{VALUE}}',
				),
				'condition' => array(
					'enable_heading_bg' => 'yes',
				),
			)
		);

		$this->add_control(
			'alignment',
			[
				'label'     => esc_html__( 'Alignment', 'studiare-plus' ),
				'type'      => 'wd_buttons',
				'options'   => [
					'left'   => [
						'title' => esc_html__( 'Left', 'studiare-plus' ),
						'image' => WOODMART_ASSETS_IMAGES . '/settings/align/left.jpg',
					],
					'center' => [
						'title' => esc_html__( 'Center', 'studiare-plus' ),
						'image' => WOODMART_ASSETS_IMAGES . '/settings/align/center.jpg',
					],
					'right'  => [
						'title' => esc_html__( 'Right', 'studiare-plus' ),
						'image' => WOODMART_ASSETS_IMAGES . '/settings/align/right.jpg',
					],
				],
				'default'   => 'center',
				'condition' => [
					'design' => 'default',
				],
			]
		);

		$this->add_control(
			'title_options_separator',
			array(
				'label'     => esc_html__( 'Title', 'studiare-plus' ),
				'type'      => Controls_Manager::HEADING,
				'separator' => 'before',
			)
		);

		$this->add_group_control(
			Group_Control_Typography::get_type(),
			array(
				'name'     => 'tabs_heading_typography',
				'label'    => esc_html__( 'Typography', 'studiare-plus' ),
				'selector' => '{{WRAPPER}} .wd-tabs .tabs-name',
			)
		);

		$this->add_control(
			'tabs_title_color',
			array(
				'label'     => esc_html__( 'Color', 'studiare-plus' ),
				'type'      => Controls_Manager::COLOR,
				'selectors' => array(
					'{{WRAPPER}} .tabs-name' => 'color: {{VALUE}}',
				),
			)
		);

		$this->add_control(
			'color',
			array(
				'label'     => esc_html__( 'Border color', 'studiare-plus' ),
				'type'      => Controls_Manager::COLOR,
				'selectors' => array(
					'{{WRAPPER}} .wd-tabs.tabs-design-simple .tabs-name' => 'border-color: {{VALUE}}',
				),
				'condition' => array(
					'design' => array( 'simple' ),
				),
			)
		);

		$this->add_control(
			'description_options_separator',
			array(
				'label'     => esc_html__( 'Description', 'studiare-plus' ),
				'type'      => Controls_Manager::HEADING,
				'separator' => 'before',
				'condition' => array(
					'design' => array( 'default', 'aside' ),
				),
			)
		);

		$this->add_control(
			'tabs_description_color',
			array(
				'label'     => esc_html__( 'Color', 'studiare-plus' ),
				'type'      => Controls_Manager::COLOR,
				'selectors' => array(
					'{{WRAPPER}} .wd-tabs-desc' => 'color: {{VALUE}}',
				),
				'condition' => array(
					'design' => array( 'default', 'aside' ),
				),
			)
		);

		$this->end_controls_section();

		$this->start_controls_section(
			'navigation_section',
			array(
				'label' => esc_html__( 'Navigation', 'studiare-plus' ),
				'tab'   => Controls_Manager::TAB_STYLE,
			)
		);

		$this->add_control(
			'tabs_style',
			array(
				'label'   => esc_html__( 'Style', 'studiare-plus' ),
				'type'    => Controls_Manager::SELECT,
				'options' => array(
					'default'   => esc_html__( 'Default', 'studiare-plus' ),
					'underline' => esc_html__( 'Underline', 'studiare-plus' ),
				),
				'default' => 'underline',
			)
		);

		$this->add_group_control(
			Group_Control_Typography::get_type(),
			array(
				'name'     => 'tabs_title_typography',
				'label'    => esc_html__( 'Typography', 'studiare-plus' ),
				'selector' => '{{WRAPPER}} .wd-nav-tabs > li > a',
			)
		);

		// Tabs start.

		$this->start_controls_tabs( 'tabs_title_text_color_tabs' );

		$this->start_controls_tab(
			'tabs_title_text_color_tab',
			array(
				'label' => esc_html__( 'Idle', 'studiare-plus' ),
			)
		);

		$this->add_control(
			'tabs_title_text_idle_color',
			array(
				'label'     => esc_html__( 'Color', 'studiare-plus' ),
				'type'      => Controls_Manager::COLOR,
				'selectors' => array(
					'{{WRAPPER}} .wd-nav-tabs' => '--nav-color: {{VALUE}}',
				),
			)
		);

		$this->add_control(
			'tabs_title_bg_color_enable',
			array(
				'label'        => esc_html__( 'Background color', 'studiare-plus' ),
				'type'         => Controls_Manager::SWITCHER,
				'default'      => '0',
				'label_on'     => esc_html__( 'Yes', 'studiare-plus' ),
				'label_off'    => esc_html__( 'No', 'studiare-plus' ),
				'return_value' => '1',
				'render_type'  => 'template',
			)
		);

		$this->add_control(
			'tabs_bg_color_idle',
			array(
				'label'     => esc_html__( 'Background color', 'studiare-plus' ),
				'type'      => Controls_Manager::COLOR,
				'selectors' => array(
					'{{WRAPPER}} .wd-nav-tabs' => '--nav-bg: {{VALUE}}',
				),
				'condition' => array(
					'tabs_title_bg_color_enable' => '1',
				),
			)
		);

		$this->add_control(
			'tabs_title_border_enable',
			array(
				'label'        => esc_html__( 'Border', 'studiare-plus' ),
				'type'         => Controls_Manager::SWITCHER,
				'default'      => '0',
				'label_on'     => esc_html__( 'Yes', 'studiare-plus' ),
				'label_off'    => esc_html__( 'No', 'studiare-plus' ),
				'return_value' => '1',
				'render_type'  => 'template',
			)
		);

		$this->add_group_control(
			Group_Control_Border::get_type(),
			array(
				'name'      => 'tabs_title_border',
				'selector'  => '{{WRAPPER}} .wd-nav-tabs > li > a',
				'condition' => array(
					'tabs_title_border_enable' => '1',
				),
			)
		);

		$this->add_responsive_control(
			'tabs_title_border_radius',
			array(
				'label'      => esc_html__( 'Border Radius', 'studiare-plus' ),
				'type'       => Controls_Manager::DIMENSIONS,
				'size_units' => array( 'px', '%', 'em' ),
				'selectors'  => array(
					'{{WRAPPER}} .wd-nav-tabs > li > a' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				),
				'condition'  => array(
					'tabs_title_border_enable' => '1',
				),
			)
		);

		$this->add_control(
			'tabs_title_box_shadow_enable',
			array(
				'label'        => esc_html__( 'Box shadow', 'studiare-plus' ),
				'type'         => Controls_Manager::SWITCHER,
				'default'      => '0',
				'label_on'     => esc_html__( 'Yes', 'studiare-plus' ),
				'label_off'    => esc_html__( 'No', 'studiare-plus' ),
				'return_value' => '1',
				'render_type'  => 'template',
			)
		);

		$this->add_group_control(
			Group_Control_Box_Shadow::get_type(),
			array(
				'name'      => 'tabs_box_shadow',
				'selector'  => '{{WRAPPER}} .wd-nav-tabs > li > a',
				'condition' => array(
					'tabs_title_box_shadow_enable' => '1',
				),
			)
		);

		$this->end_controls_tab();

		$this->start_controls_tab(
			'tabs_title_text_hover_color_tab',
			array(
				'label' => esc_html__( 'Hover', 'studiare-plus' ),
			)
		);

		$this->add_control(
			'tabs_title_text_hover_color',
			array(
				'label'     => esc_html__( 'Color', 'studiare-plus' ),
				'type'      => Controls_Manager::COLOR,
				'selectors' => array(
					'{{WRAPPER}} .wd-nav-tabs' => '--nav-color-hover: {{VALUE}}',
				),
			)
		);

		$this->add_control(
			'tabs_title_bg_hover_color_enable',
			array(
				'label'        => esc_html__( 'Background color', 'studiare-plus' ),
				'type'         => Controls_Manager::SWITCHER,
				'default'      => '0',
				'label_on'     => esc_html__( 'Yes', 'studiare-plus' ),
				'label_off'    => esc_html__( 'No', 'studiare-plus' ),
				'return_value' => '1',
				'render_type'  => 'template',
			)
		);

		$this->add_control(
			'tabs_bg_color_hover',
			array(
				'label'     => esc_html__( 'Background color', 'studiare-plus' ),
				'type'      => Controls_Manager::COLOR,
				'selectors' => array(
					'{{WRAPPER}} .wd-nav-tabs' => '--nav-bg-hover: {{VALUE}}',
				),
				'condition' => array(
					'tabs_title_bg_hover_color_enable' => '1',
				),
			)
		);

		$this->add_control(
			'tabs_title_border_hover_enable',
			array(
				'label'        => esc_html__( 'Border', 'studiare-plus' ),
				'type'         => Controls_Manager::SWITCHER,
				'default'      => '0',
				'label_on'     => esc_html__( 'Yes', 'studiare-plus' ),
				'label_off'    => esc_html__( 'No', 'studiare-plus' ),
				'return_value' => '1',
				'render_type'  => 'template',
			)
		);

		$this->add_group_control(
			Group_Control_Border::get_type(),
			array(
				'name'      => 'tabs_border_hover',
				'selector'  => '{{WRAPPER}} .wd-nav-tabs > li:hover > a',
				'condition' => array(
					'tabs_title_border_hover_enable' => '1',
				),
			)
		);

		$this->add_control(
			'tabs_title_box_shadow_hover_enable',
			array(
				'label'        => esc_html__( 'Box shadow', 'studiare-plus' ),
				'type'         => Controls_Manager::SWITCHER,
				'default'      => '0',
				'label_on'     => esc_html__( 'Yes', 'studiare-plus' ),
				'label_off'    => esc_html__( 'No', 'studiare-plus' ),
				'return_value' => '1',
				'render_type'  => 'template',
			)
		);

		$this->add_group_control(
			Group_Control_Box_Shadow::get_type(),
			array(
				'name'      => 'tabs_box_shadow_hover',
				'selector'  => '{{WRAPPER}} .wd-nav-tabs > li:hover > a',
				'condition' => array(
					'tabs_title_box_shadow_hover_enable' => '1',
				),
			)
		);

		$this->end_controls_tab();

		$this->start_controls_tab(
			'tabs_title_text_hover_active_tab',
			array(
				'label' => esc_html__( 'Active', 'studiare-plus' ),
			)
		);

		$this->add_control(
			'tabs_title_text_hover_active',
			array(
				'label'     => esc_html__( 'Color', 'studiare-plus' ),
				'type'      => Controls_Manager::COLOR,
				'selectors' => array(
					'{{WRAPPER}} .wd-nav-tabs' => '--nav-color-active: {{VALUE}}',
				),
			)
		);

		$this->add_control(
			'tabs_title_bg_active_color_enable',
			array(
				'label'        => esc_html__( 'Background color', 'studiare-plus' ),
				'type'         => Controls_Manager::SWITCHER,
				'default'      => '0',
				'label_on'     => esc_html__( 'Yes', 'studiare-plus' ),
				'label_off'    => esc_html__( 'No', 'studiare-plus' ),
				'return_value' => '1',
				'render_type'  => 'template',
			)
		);

		$this->add_control(
			'tabs_bg_color_active',
			array(
				'label'     => esc_html__( 'Background color', 'studiare-plus' ),
				'type'      => Controls_Manager::COLOR,
				'selectors' => array(
					'{{WRAPPER}} .wd-nav-tabs' => '--nav-bg-active: {{VALUE}}',
				),
				'condition' => array(
					'tabs_title_bg_active_color_enable' => '1',
				),
			)
		);

		$this->add_control(
			'tabs_title_border_active_enable',
			array(
				'label'        => esc_html__( 'Border', 'studiare-plus' ),
				'type'         => Controls_Manager::SWITCHER,
				'default'      => '0',
				'label_on'     => esc_html__( 'Yes', 'studiare-plus' ),
				'label_off'    => esc_html__( 'No', 'studiare-plus' ),
				'return_value' => '1',
				'render_type'  => 'template',
			)
		);

		$this->add_group_control(
			Group_Control_Border::get_type(),
			array(
				'name'      => 'tabs_border_active',
				'selector'  => '{{WRAPPER}} .wd-nav-tabs > li.wd-active > a',
				'condition' => array(
					'tabs_title_border_active_enable' => '1',
				),
			)
		);

		$this->add_control(
			'tabs_title_box_shadow_active_enable',
			array(
				'label'        => esc_html__( 'Box shadow', 'studiare-plus' ),
				'type'         => Controls_Manager::SWITCHER,
				'default'      => '0',
				'label_on'     => esc_html__( 'Yes', 'studiare-plus' ),
				'label_off'    => esc_html__( 'No', 'studiare-plus' ),
				'return_value' => '1',
				'render_type'  => 'template',
			)
		);

		$this->add_group_control(
			Group_Control_Box_Shadow::get_type(),
			array(
				'name'      => 'tabs_box_shadow_active',
				'selector'  => '{{WRAPPER}} .wd-nav-tabs > li.wd-active > a',
				'condition' => array(
					'tabs_title_box_shadow_active_enable' => '1',
				),
			)
		);

		$this->end_controls_tab();

		$this->end_controls_tabs();

		// Tabs end.

		$this->add_control(
			'items_separator',
			array(
				'type' => Controls_Manager::DIVIDER,
			)
		);

		$this->add_responsive_control(
			'space_between_tabs_title_horizontal',
			array(
				'label'     => esc_html__( 'Items gap', 'studiare-plus' ),
				'type'      => Controls_Manager::SLIDER,
				'range'     => array(
					'px' => array(
						'min'  => 0,
						'max'  => 150,
						'step' => 1,
					),
				),
				'selectors' => array(
					'{{WRAPPER}} .wd-nav-tabs' => '--nav-gap: {{SIZE}}px;',
				),
			)
		);

		$this->add_responsive_control(
			'tabs_title_padding',
			array(
				'label'      => esc_html__( 'Padding', 'studiare-plus' ),
				'type'       => Controls_Manager::DIMENSIONS,
				'size_units' => array( 'px', '%', 'em' ),
				'selectors'  => array(
					'{{WRAPPER}} .wd-nav-tabs' => '--nav-pd: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				),
			)
		);

		$this->add_control(
			'title_text_color_scheme',
			array(
				'label'   => esc_html__( 'Color scheme', 'studiare-plus' ),
				'type'    => Controls_Manager::SELECT,
				'options' => array(
					'inherit' => esc_html__( 'Inherit', 'studiare-plus' ),
					'light'   => esc_html__( 'Light', 'studiare-plus' ),
					'dark'    => esc_html__( 'Dark', 'studiare-plus' ),
				),
				'default' => 'inherit',
			)
		);

		$this->add_control(
			'icon_options_separator',
			array(
				'label'     => esc_html__( 'Icon', 'studiare-plus' ),
				'type'      => Controls_Manager::HEADING,
				'separator' => 'before',
			)
		);

		$this->add_control(
			'icon_alignment_design_default',
			array(
				'label'     => esc_html__( 'Icon alignment', 'studiare-plus' ),
				'type'      => 'wd_buttons',
				'options'   => array(
					'left'  => array(
						'title' => esc_html__( 'Left', 'studiare-plus' ),
						'image' => WOODMART_ASSETS_IMAGES . '/settings/infobox/position/left.png',
					),
					'top'   => array(
						'title' => esc_html__( 'Top', 'studiare-plus' ),
						'image' => WOODMART_ASSETS_IMAGES . '/settings/infobox/position/top.png',
					),
					'right' => array(
						'title' => esc_html__( 'Right', 'studiare-plus' ),
						'image' => WOODMART_ASSETS_IMAGES . '/settings/infobox/position/right.png',
					),
				),
				'default'   => 'top',
				'condition' => array(
					'design' => array( 'default' ),
				),
			)
		);

		$this->add_control(
			'icon_alignment',
			array(
				'label'     => esc_html__( 'Alignment', 'studiare-plus' ),
				'type'      => 'wd_buttons',
				'options'   => array(
					'left'  => array(
						'title' => esc_html__( 'Left', 'studiare-plus' ),
						'image' => WOODMART_ASSETS_IMAGES . '/settings/infobox/position/left.png',
					),
					'top'   => array(
						'title' => esc_html__( 'Top', 'studiare-plus' ),
						'image' => WOODMART_ASSETS_IMAGES . '/settings/infobox/position/top.png',
					),
					'right' => array(
						'title' => esc_html__( 'Right', 'studiare-plus' ),
						'image' => WOODMART_ASSETS_IMAGES . '/settings/infobox/position/right.png',
					),
				),
				'default'   => 'left',
				'condition' => array(
					'design!' => array( 'default' ),
				),
			)
		);

		$this->end_controls_section();

		/**
		 * Products layout settings.
		 */
		$this->start_controls_section(
			'products_layout_style_section',
			[
				'label' => esc_html__( 'Products layout', 'studiare-plus' ),
				'tab'   => Controls_Manager::TAB_STYLE,
			]
		);

		$this->add_control(
			'layout',
			[
				'label'   => esc_html__( 'Layout', 'studiare-plus' ),
				'type'    => Controls_Manager::SELECT,
				'default' => 'grid',
				'options' => array(
					'grid'     => esc_html__( 'Grid', 'studiare-plus' ),
					'list'     => esc_html__( 'List', 'studiare-plus' ),
					'carousel' => esc_html__( 'Carousel', 'studiare-plus' ),
				),
			]
		);

		$this->add_responsive_control(
			'columns',
			[
				'label'       => esc_html__( 'Columns', 'studiare-plus' ),
				'description' => esc_html__( 'Number of columns in the grid.', 'studiare-plus' ),
				'type'        => Controls_Manager::SLIDER,
				'default'     => [
					'size' => 4,
				],
				'size_units'  => '',
				'range'       => [
					'px' => [
						'min'  => 1,
						'max'  => 6,
						'step' => 1,
					],
				],
				'devices'     => array( 'desktop', 'tablet', 'mobile' ),
				'classes'     => 'wd-hide-custom-breakpoints',
				'condition'   => [
					'layout' => 'grid',
				],
			]
		);

		$this->add_control(
			'products_masonry',
			[
				'label'       => esc_html__( 'Masonry grid', 'studiare-plus' ),
				'description' => esc_html__( 'Products may have different sizes.', 'studiare-plus' ),
				'type'        => Controls_Manager::SELECT,
				'default'     => '',
				'options'     => array(
					''        => esc_html__( 'Inherit', 'studiare-plus' ),
					'enable'  => esc_html__( 'Enable', 'studiare-plus' ),
					'disable' => esc_html__( 'Disable', 'studiare-plus' ),
				),
				'condition'   => [
					'layout' => 'grid',
				],
			]
		);

		$this->add_control(
			'products_different_sizes',
			[
				'label'       => esc_html__( 'Products grid with different sizes', 'studiare-plus' ),
				'description' => esc_html__( 'In this situation, some of the products will be twice bigger in width than others. Recommended to use with 6 columns grid only.', 'studiare-plus' ),
				'type'        => Controls_Manager::SELECT,
				'default'     => '',
				'options'     => array(
					''        => esc_html__( 'Inherit', 'studiare-plus' ),
					'enable'  => esc_html__( 'Enable', 'studiare-plus' ),
					'disable' => esc_html__( 'Disable', 'studiare-plus' ),
				),
				'condition'   => [
					'layout' => 'grid',
				],
			]
		);

		$this->add_responsive_control(
			'spacing',
			[
				'label'     => esc_html__( 'Space between', 'studiare-plus' ),
				'type'      => Controls_Manager::SELECT,
				'options'   => [
					'' => esc_html__( 'Inherit', 'studiare-plus' ),
					0  => esc_html__( '0 px', 'studiare-plus' ),
					2  => esc_html__( '2 px', 'studiare-plus' ),
					6  => esc_html__( '6 px', 'studiare-plus' ),
					10 => esc_html__( '10 px', 'studiare-plus' ),
					20 => esc_html__( '20 px', 'studiare-plus' ),
					30 => esc_html__( '30 px', 'studiare-plus' ),
				],
				'default'   => '',
				'devices'   => array( 'desktop', 'tablet', 'mobile' ),
				'classes'   => 'wd-hide-custom-breakpoints',
				'condition' => [
					'layout' => [ 'grid', 'carousel' ],
				],
			]
		);

		$this->add_control(
			'items_per_page',
			[
				'label'       => esc_html__( 'Items per page', 'studiare-plus' ),
				'description' => esc_html__( 'Number of items to show per page.', 'studiare-plus' ),
				'default'     => 12,
				'type'        => Controls_Manager::NUMBER,
			]
		);

		$this->add_control(
			'pagination',
			[
				'label'     => esc_html__( 'Pagination', 'studiare-plus' ),
				'type'      => Controls_Manager::SELECT,
				'default'   => '',
				'options'   => array(
					''         => esc_html__( 'Inherit', 'studiare-plus' ),
					'more-btn' => esc_html__( 'Load more button', 'studiare-plus' ),
					'infinit'  => esc_html__( 'Infinit scrolling', 'studiare-plus' ),
					'arrows'   => esc_html__( 'Arrows', 'studiare-plus' ),
				),
				'condition' => [
					'layout!' => 'carousel',
				],
			]
		);

		$this->add_control(
			'pagination_arrows_position_popover',
			array(
				'label'     => esc_html__( 'Arrows position', 'studiare-plus' ),
				'type'      => Controls_Manager::POPOVER_TOGGLE,
				'condition' => array(
					'layout!'    => 'carousel',
					'pagination' => 'arrows',
				),
			)
		);

		$this->start_popover();

		$this->add_control(
			'pagination_arrows_position',
			[
				'label'     => esc_html__( 'Position', 'studiare-plus' ),
				'type'      => Controls_Manager::SELECT,
				'default'   => '',
				'options'   => array(
					''         => esc_html__( 'Inherit from Theme Settings', 'studiare-plus' ),
					'sep'      => esc_html__( 'Separate', 'studiare-plus' ),
					'together' => esc_html__( 'Together', 'studiare-plus' ),
				),
				'condition' => [
					'layout!'                             => 'carousel',
					'pagination'                          => 'arrows',
					'pagination_arrows_position_popover!' => '',
				],
			]
		);

		$this->add_responsive_control(
			'pagination_arrows_offset_h',
			array(
				'label'      => esc_html__( 'Offset horizontal', 'studiare-plus' ),
				'type'       => Controls_Manager::SLIDER,
				'size_units' => array( 'px' ),
				'range'      => array(
					'px' => array(
						'min'  => -500,
						'max'  => 500,
						'step' => 1,
					),
				),
				'selectors'  => array(
					'{{WRAPPER}} .wd-nav-arrows' => '--wd-arrow-offset-h: {{SIZE}}{{UNIT}};',
				),
				'condition'  => array(
					'layout!'                             => 'carousel',
					'pagination'                          => 'arrows',
					'pagination_arrows_position_popover!' => '',
				),
			)
		);

		$this->add_responsive_control(
			'pagination_arrows_offset_v',
			array(
				'label'      => esc_html__( 'Offset vertical', 'studiare-plus' ),
				'type'       => Controls_Manager::SLIDER,
				'size_units' => array( 'px' ),
				'range'      => array(
					'px' => array(
						'min'  => -500,
						'max'  => 500,
						'step' => 1,
					),
				),
				'selectors'  => array(
					'{{WRAPPER}} .wd-nav-arrows' => '--wd-arrow-offset-v: {{SIZE}}{{UNIT}};',
				),
				'condition'  => array(
					'layout!'                             => 'carousel',
					'pagination'                          => 'arrows',
					'pagination_arrows_position_popover!' => '',
				),
			)
		);

		$this->end_popover();

		$this->end_controls_section();

		/**
		 * Products design settings.
		 */
		$this->start_controls_section(
			'products_design_style_section',
			[
				'label' => esc_html__( 'Products design', 'studiare-plus' ),
				'tab'   => Controls_Manager::TAB_STYLE,
			]
		);

		$this->add_control(
			'product_hover',
			[
				'label'     => esc_html__( 'Products hover', 'studiare-plus' ),
				'type'      => Controls_Manager::SELECT,
				'default'   => 'inherit',
				'options'   => array(
					'inherit'          => esc_html__( 'Inherit from Theme Settings', 'studiare-plus' ),
					'info-alt'         => esc_html__( 'Full info on hover', 'studiare-plus' ),
					'info'             => esc_html__( 'Full info on image', 'studiare-plus' ),
					'alt'              => esc_html__( 'Icons and "add to cart" on hover', 'studiare-plus' ),
					'icons'            => esc_html__( 'Icons on hover', 'studiare-plus' ),
					'quick'            => esc_html__( 'Quick', 'studiare-plus' ),
					'button'           => esc_html__( 'Show button on hover on image', 'studiare-plus' ),
					'base'             => esc_html__( 'Show summary on hover', 'studiare-plus' ),
					'standard'         => esc_html__( 'Standard button', 'studiare-plus' ),
					'tiled'            => esc_html__( 'Tiled', 'studiare-plus' ),
					'fw-button'        => esc_html__( 'Full width button', 'studiare-plus' ),
					'small-icon'       => esc_html__( 'آیکون کوچک 80*80', 'studiare-plus' ),
					'old-stud'         => esc_html__( 'استایل کلاسیک استادیار', 'studiare-plus' ),
					'zhaket'           => esc_html__( 'کارت محصول ژاکت', 'studiare-plus' ),
					'buttons-on-hover' => esc_html__( 'Buttons on hover', 'studiare-plus' ),
				),
				'condition' => [
					'layout!' => 'list',
				],
			]
		);

		$this->add_control(
			'img_size',
			[
				'label'   => esc_html__( 'Image size', 'studiare-plus' ),
				'type'    => Controls_Manager::SELECT,
				'default' => 'large',
				'options' => woodmart_get_all_image_sizes_names( 'elementor' ),
			]
		);

		$this->add_control(
			'img_size_custom',
			[
				'label'       => esc_html__( 'Image dimension', 'studiare-plus' ),
				'type'        => Controls_Manager::IMAGE_DIMENSIONS,
				'description' => esc_html__( 'You can crop the original image size to any custom size. You can also set a single value for height or width in order to keep the original size ratio.', 'studiare-plus' ),
				'condition'   => [
					'img_size' => 'custom',
				],
			]
		);

		$this->add_control(
			'rounding_size',
			array(
				'label'     => esc_html__( 'Rounding', 'studiare-plus' ),
				'type'      => Controls_Manager::SELECT,
				'options'   => array(
					''       => esc_html__( 'Inherit', 'studiare-plus' ),
					'0'      => esc_html__( '0', 'studiare-plus' ),
					'5'      => esc_html__( '5', 'studiare-plus' ),
					'8'      => esc_html__( '8', 'studiare-plus' ),
					'12'     => esc_html__( '12', 'studiare-plus' ),
					'custom' => esc_html__( 'Custom', 'studiare-plus' ),
				),
				'default'   => '',
				'selectors' => array(
					'{{WRAPPER}}' => '--wd-brd-radius: {{VALUE}}px;',
				),
			)
		);

		$this->add_control(
			'custom_rounding_size',
			array(
				'label'      => esc_html__( 'Rounding', 'studiare-plus' ),
				'type'       => Controls_Manager::SLIDER,
				'size_units' => array( '%', 'px' ),
				'range'      => array(
					'px' => array(
						'min'  => 1,
						'max'  => 300,
						'step' => 1,
					),
					'%'  => array(
						'min'  => 1,
						'max'  => 100,
						'step' => 1,
					),
				),
				'selectors'  => array(
					'{{WRAPPER}}' => '--wd-brd-radius: {{SIZE}}{{UNIT}};',
				),
				'condition'  => array(
					'rounding_size' => array( 'custom' ),
				),
			)
		);

		$this->add_control(
			'sale_countdown',
			[
				'label'        => esc_html__( 'Sale countdown', 'studiare-plus' ),
				'description'  => esc_html__( 'Countdown to the end sale date will be shown. Be sure you have set final date of the product sale price.', 'studiare-plus' ),
				'type'         => Controls_Manager::SWITCHER,
				'default'      => '0',
				'label_on'     => esc_html__( 'Yes', 'studiare-plus' ),
				'label_off'    => esc_html__( 'No', 'studiare-plus' ),
				'return_value' => '1',
			]
		);

		$this->add_responsive_control(
			'stretch_product',
			[
				'label'        => esc_html__( 'Even product grid', 'studiare-plus' ),
				'type'         => Controls_Manager::SWITCHER,
				'default'      => '0',
				'label_on'     => esc_html__( 'Yes', 'studiare-plus' ),
				'label_off'    => esc_html__( 'No', 'studiare-plus' ),
				'return_value' => '1',
				'devices'      => array( 'desktop', 'tablet', 'mobile' ),
				'classes'      => 'wd-hide-custom-breakpoints',
				'condition'    => array(
					'product_hover' => array( 'icons', 'alt', 'button', 'standard', 'tiled', 'quick', 'base', 'fw-button', 'buttons-on-hover' ),
				),
			]
		);

		$this->add_control(
			'stock_progress_bar',
			[
				'label'        => esc_html__( 'Stock progress bar', 'studiare-plus' ),
				'description'  => esc_html__( 'Display a number of sold and in stock products as a progress bar.', 'studiare-plus' ),
				'type'         => Controls_Manager::SWITCHER,
				'default'      => '0',
				'label_on'     => esc_html__( 'Yes', 'studiare-plus' ),
				'label_off'    => esc_html__( 'No', 'studiare-plus' ),
				'return_value' => '1',
			]
		);

		$this->add_control(
			'products_color_scheme',
			array(
				'label'        => esc_html__( 'Products color scheme', 'studiare-plus' ),
				'type'         => Controls_Manager::SELECT,
				'default'      => 'default',
				'options'      => array(
					'default' => esc_html__( 'Default', 'studiare-plus' ),
					'dark'    => esc_html__( 'Dark', 'studiare-plus' ),
					'light'   => esc_html__( 'Light', 'studiare-plus' ),
				),
			)
		);

		$this->add_control(
			'products_divider',
			[
				'label'        => esc_html__( 'Products divider', 'studiare-plus' ),
				'type'         => Controls_Manager::SWITCHER,
				'default'      => '0',
				'label_on'     => esc_html__( 'Yes', 'studiare-plus' ),
				'label_off'    => esc_html__( 'No', 'studiare-plus' ),
				'return_value' => '1',
				'condition'    => [
					'product_hover' => [ 'small' ],
					'layout'        => 'grid',
				],
			]
		);

		$this->add_control(
			'products_bordered_grid',
			[
				'label'        => esc_html__( 'Products border', 'studiare-plus' ),
				'description'  => esc_html__( 'Add borders between the products in your grid.', 'studiare-plus' ),
				'type'         => Controls_Manager::SWITCHER,
				'default'      => '0',
				'label_on'     => esc_html__( 'Yes', 'studiare-plus' ),
				'label_off'    => esc_html__( 'No', 'studiare-plus' ),
				'return_value' => '1',
			]
		);

		$this->add_control(
			'products_bordered_grid_style',
			array(
				'label'     => esc_html__( 'Border position', 'studiare-plus' ),
				'type'      => Controls_Manager::SELECT,
				'options'   => array(
					'outside' => esc_html__( 'Outside', 'studiare-plus' ),
					'inside'  => esc_html__( 'Inside', 'studiare-plus' ),
				),
				'default'   => 'outside',
				'condition' => array(
					'products_bordered_grid' => array( '1' ),
				),
			)
		);

		$this->add_control(
			'products_border_color',
			array(
				'label'     => esc_html__( 'Custom border color', 'studiare-plus' ),
				'type'      => Controls_Manager::COLOR,
				'selectors' => array(
					'{{WRAPPER}} [class*="products-bordered-grid"], {{WRAPPER}} [class*="products-bordered-grid"] .wd-product' => '--wd-bordered-brd:{{VALUE}};',
				),
				'condition' => array(
					'products_bordered_grid' => array( '1' ),
				),
			)
		);

		$this->add_control(
			'products_with_background',
			array(
				'label'        => esc_html__( 'Products background', 'studiare-plus' ),
				'description'  => esc_html__( 'Add a background to the products in your grid.', 'studiare-plus' ),
				'type'         => Controls_Manager::SWITCHER,
				'default'      => '0',
				'label_on'     => esc_html__( 'Yes', 'studiare-plus' ),
				'label_off'    => esc_html__( 'No', 'studiare-plus' ),
				'return_value' => '1',
			)
		);

		$this->add_control(
			'products_background',
			array(
				'label'     => esc_html__( 'Custom background color', 'studiare-plus' ),
				'type'      => Controls_Manager::COLOR,
				'selectors' => array(
					'{{WRAPPER}} .wd-products-with-bg, {{WRAPPER}} .wd-products-with-bg .wd-product' => '--wd-prod-bg:{{VALUE}}; --wd-bordered-bg:{{VALUE}};',
				),
				'condition' => array(
					'products_with_background' => array( '1' ),
				),
			)
		);

		$this->add_control(
			'products_shadow',
			array(
				'label'        => esc_html__( 'Products shadow', 'studiare-plus' ),
				'description'  => esc_html__( 'Add a shadow to products if the initial product style did not have one.', 'studiare-plus' ),
				'type'         => Controls_Manager::SWITCHER,
				'default'      => '0',
				'label_on'     => esc_html__( 'Yes', 'studiare-plus' ),
				'label_off'    => esc_html__( 'No', 'studiare-plus' ),
				'return_value' => '1',
			)
		);

		$this->add_control(
			'product_quantity',
			[
				'label'        => esc_html__( 'Quantity input on product', 'studiare-plus' ),
				'type'        => Controls_Manager::SELECT,
				'default'     => '',
				'options'     => array(
					''        => esc_html__( 'Inherit', 'studiare-plus' ),
					'enable'  => esc_html__( 'Enable', 'studiare-plus' ),
					'disable' => esc_html__( 'Disable', 'studiare-plus' ),
				),
			]
		);

		$this->add_control(
			'grid_gallery',
			array(
				'label'       => esc_html__( 'Product gallery', 'studiare-plus' ),
				'description' => esc_html__( 'Add the ability to view the product gallery on the products loop.', 'studiare-plus' ),
				'type'        => Controls_Manager::SELECT,
				'default'     => '',
				'options'     => array(
					''    => esc_html__( 'Inherit', 'studiare-plus' ),
					'yes' => esc_html__( 'Enable', 'studiare-plus' ),
					'no'  => esc_html__( 'Disable', 'studiare-plus' ),
				),
			)
		);

		$this->start_controls_tabs(
			'grid_gallery_tabs',
			array(
				'condition' => array(
					'grid_gallery' => array( 'yes' ),
				),
			)
		);

		$this->start_controls_tab(
			'grid_gallery_desktop_tab',
			array(
				'label' => esc_html__( 'Desktop', 'studiare-plus' ),
			)
		);

		$this->add_control(
			'grid_gallery_control',
			array(
				'label'   => esc_html__( 'Product gallery controls', 'studiare-plus' ),
				'type'    => Controls_Manager::SELECT,
				'default' => '',
				'options' => array(
					''       => esc_html__( 'Inherit', 'studiare-plus' ),
					'arrows' => esc_html__( 'Arrows', 'studiare-plus' ),
					'hover'  => esc_html__( 'Hover', 'studiare-plus' ),
				),
			)
		);

		$this->end_controls_tab();

		$this->start_controls_tab(
			'grid_gallery_mobile_tab',
			[
				'label' => esc_html__( 'Mobile devices', 'studiare-plus' ),
			]
		);

		$this->add_control(
			'grid_gallery_enable_arrows',
			array(
				'label'   => esc_html__( 'Product gallery controls', 'studiare-plus' ),
				'type'    => Controls_Manager::SELECT,
				'default' => '',
				'options' => array(
					''       => esc_html__( 'Inherit', 'studiare-plus' ),
					'none'   => esc_html__( 'None', 'studiare-plus' ),
					'arrows' => esc_html__( 'Arrows', 'studiare-plus' ),
				),
			)
		);

		$this->end_controls_tab();

		$this->end_controls_tabs();

		$this->end_controls_section();

		/**
		 * Extra settings.
		 */
		$this->start_controls_section(
			'extra_style_section',
			[
				'label' => esc_html__( 'Extra', 'studiare-plus' ),
				'tab'   => Controls_Manager::TAB_STYLE,
			]
		);

		$this->add_control(
			'lazy_loading',
			[
				'label'        => esc_html__( 'Lazy loading for images', 'studiare-plus' ),
				'description'  => esc_html__( 'Enable lazy loading for images for this element.', 'studiare-plus' ),
				'type'         => Controls_Manager::SWITCHER,
				'default'      => 'no',
				'label_on'     => esc_html__( 'Yes', 'studiare-plus' ),
				'label_off'    => esc_html__( 'No', 'studiare-plus' ),
				'return_value' => 'yes',
			]
		);

		$this->end_controls_section();
	}

	/**
	 * Render the widget output on the frontend.
	 *
	 * Written in PHP and used to generate the final HTML.
	 *
	 * @since 1.0.0
	 *
	 * @access protected
	 */
	protected function render() {
		woodmart_elementor_products_tabs_template( $this->get_settings_for_display() );
	}
}

Plugin::instance()->widgets_manager->register( new Products_Tabs() );
