<?php
/**
 * Instagram map.
 */

namespace XTS\Elementor;

use Elementor\Group_Control_Typography;
use Elementor\Widget_Base;
use Elementor\Controls_Manager;
use Elementor\Plugin;

if ( ! defined( 'ABSPATH' ) ) {
	exit; // Direct access not allowed.
}

/**
 * Elementor widget that inserts an embeddable content into the page, from any given URL.
 *
 * @since 1.0.0
 */
class Instagram extends Widget_Base {
	/**
	 * Get widget name.
	 *
	 * @since  1.0.0
	 * @access public
	 *
	 * @return string Widget name.
	 */
	public function get_name() {
		return 'wd_instagram';
	}

	/**
	 * Get widget title.
	 *
	 * @since  1.0.0
	 * @access public
	 *
	 * @return string Widget title.
	 */
	public function get_title() {
		return esc_html__( 'Instagram', 'studiare-plus' );
	}

	/**
	 * Get widget icon.
	 *
	 * @since  1.0.0
	 * @access public
	 *
	 * @return string Widget icon.
	 */
	public function get_icon() {
		return 'wd-icon-instagram';
	}

	/**
	 * Get widget categories.
	 *
	 * @since  1.0.0
	 * @access public
	 *
	 * @return array Widget categories.
	 */
	public function get_categories() {
		return [ 'wd-elements' ];
	}

	/**
	 * Register the widget controls.
	 *
	 * @since  1.0.0
	 * @access protected
	 */
	protected function register_controls() {
		/**
		 * Content tab.
		 */

		/**
		 * General settings.
		 */
		$this->start_controls_section(
			'general_content_section',
			[
				'label' => esc_html__( 'General', 'studiare-plus' ),
			]
		);

		$this->add_control(
			'extra_width_classes',
			array(
				'type'         => 'wd_css_class',
				'default'      => 'wd-width-100',
				'prefix_class' => '',
			)
		);

		$this->add_control(
			'data_source',
			[
				'label'       => esc_html__( 'Source type', 'studiare-plus' ),
				'description' => 'API request type<br>
Scrape - parse Instagram page and take photos by username. Now deprecated and may be blocked by Instagram.<br>
API - the best safe and legal option to obtain Instagram photos. Requires Instagram APP configuration. <br>
Follow our documentation <a href="https://xtemos.com/docs/woodmart/faq-guides/setup-instagram-api/" target="_blank">here</a>',
				'type'        => Controls_Manager::SELECT,
				'options'     => [
					'scrape' => esc_html__( 'Scrape (deprecated)', 'studiare-plus' ),
					'api'    => esc_html__( 'API', 'studiare-plus' ),
					'images' => esc_html__( 'Images', 'studiare-plus' ),
				],
				'default'     => 'images',
			]
		);

		$this->add_control(
			'size',
			[
				'label'       => esc_html__( 'Photo size', 'studiare-plus' ),
				'description' => esc_html__( 'Enter image size. Example: \'thumbnail\', \'medium\', \'large\', \'full\'. Leave empty to use \'medium\' size.', 'studiare-plus' ),
				'type'        => Controls_Manager::TEXT,
				'default'     => 'medium',
				'condition'   => [
					'data_source!' => [ 'images' ],
				],
			]
		);

		$this->end_controls_section();

		/**
		 * Images settings.
		 */
		$this->start_controls_section(
			'images_section',
			[
				'label'     => esc_html__( 'Images', 'studiare-plus' ),
				'condition' => [
					'data_source' => [ 'images' ],
				],
			]
		);

		$this->add_control(
			'images',
			[
				'label'   => esc_html__( 'Images', 'studiare-plus' ),
				'type'    => Controls_Manager::GALLERY,
				'default' => [],
			]
		);

		$this->add_control(
			'images_size',
			[
				'label'       => esc_html__( 'Images size', 'studiare-plus' ),
				'description' => esc_html__( 'Enter image size. Example: \'thumbnail\', \'medium\', \'large\', \'full\'. Leave empty to use \'thumbnail\' size.', 'studiare-plus' ),
				'type'        => Controls_Manager::TEXT,
				'default'     => 'thumbnail',
			]
		);

		$this->add_control(
			'images_link',
			[
				'label' => esc_html__( 'Images link', 'studiare-plus' ),
				'type'  => Controls_Manager::TEXT,
			]
		);

		$this->add_control(
			'images_likes',
			[
				'label'       => esc_html__( 'Likes limit', 'studiare-plus' ),
				'description' => esc_html__( 'Example: 1000-10000', 'studiare-plus' ),
				'type'        => Controls_Manager::TEXT,
				'default'     => '1000-10000',
			]
		);

		$this->add_control(
			'images_comments',
			[
				'label'       => esc_html__( 'Comments limit', 'studiare-plus' ),
				'description' => esc_html__( 'Example: 0-1000', 'studiare-plus' ),
				'type'        => Controls_Manager::TEXT,
				'default'     => '0-1000',
			]
		);

		$this->end_controls_section();

		/**
		 * Content settings.
		 */
		$this->start_controls_section(
			'content_section',
			[
				'label' => esc_html__( 'Content', 'studiare-plus' ),
			]
		);

		$this->add_control(
			'content',
			[
				'label' => esc_html__( 'Instagram text', 'studiare-plus' ),
				'type'  => Controls_Manager::WYSIWYG,
			]
		);

		$this->end_controls_section();

		/**
		 * Link settings.
		 */
		$this->start_controls_section(
			'link_section',
			[
				'label' => esc_html__( 'Link', 'studiare-plus' ),
			]
		);

		$this->add_control(
			'username',
			[
				'label'   => esc_html__( 'Username', 'studiare-plus' ),
				'type'    => Controls_Manager::TEXT,
				'default' => 'flickr',
			]
		);

		$this->add_control(
			'link',
			[
				'label' => esc_html__( 'Link text', 'studiare-plus' ),
				'type'  => Controls_Manager::TEXT,
			]
		);

		$this->add_control(
			'target',
			[
				'label'   => esc_html__( 'Open link in', 'studiare-plus' ),
				'type'    => Controls_Manager::SELECT,
				'options' => [
					'_self'  => esc_html__( 'Current window (_self)', 'studiare-plus' ),
					'_blank' => esc_html__( 'New window (_blank)', 'studiare-plus' ),
				],
				'default' => '_self',
			]
		);

		$this->end_controls_section();

		/**
		 * Style tab.
		 */

		/**
		 * Layout settings.
		 */
		$this->start_controls_section(
			'layout_content_section',
			[
				'label' => esc_html__( 'Layout', 'studiare-plus' ),
				'tab'   => Controls_Manager::TAB_STYLE,
			]
		);

		$this->add_control(
			'design',
			[
				'label'   => esc_html__( 'Layout', 'studiare-plus' ),
				'type'    => Controls_Manager::SELECT,
				'options' => [
					'grid'   => esc_html__( 'Grid', 'studiare-plus' ),
					'slider' => esc_html__( 'Carousel', 'studiare-plus' ),
				],
				'default' => 'grid',
			]
		);

		$this->add_responsive_control(
			'per_row',
			[
				'label'      => esc_html__( 'Photos per row', 'studiare-plus' ),
				'type'       => Controls_Manager::SLIDER,
				'default'    => [
					'size' => 3,
				],
				'size_units' => '',
				'range'      => [
					'px' => [
						'min'  => 1,
						'max'  => 12,
						'step' => 1,
					],
				],
				'devices'    => array( 'desktop', 'tablet', 'mobile' ),
				'classes'    => 'wd-hide-custom-breakpoints',
			]
		);

		$this->add_control(
			'number',
			[
				'label'      => esc_html__( 'Number of photos', 'studiare-plus' ),
				'type'       => Controls_Manager::SLIDER,
				'default'    => [
					'size' => 9,
				],
				'size_units' => '',
				'range'      => [
					'px' => [
						'min'  => 1,
						'max'  => 25,
						'step' => 1,
					],
				],
				'condition'  => array(
					'data_source!' => 'images',
				),
			]
		);

		$this->add_control(
			'spacing',
			[
				'label'        => esc_html__( 'Add spaces between photos', 'studiare-plus' ),
				'type'         => Controls_Manager::SWITCHER,
				'default'      => '0',
				'label_on'     => esc_html__( 'Yes', 'studiare-plus' ),
				'label_off'    => esc_html__( 'No', 'studiare-plus' ),
				'return_value' => '1',
			]
		);

		$this->add_responsive_control(
			'spacing_custom',
			[
				'label'     => esc_html__( 'Space between', 'studiare-plus' ),
				'type'      => Controls_Manager::SELECT,
				'options'   => [
					0  => esc_html__( '0 px', 'studiare-plus' ),
					2  => esc_html__( '2 px', 'studiare-plus' ),
					6  => esc_html__( '6 px', 'studiare-plus' ),
					10 => esc_html__( '10 px', 'studiare-plus' ),
					20 => esc_html__( '20 px', 'studiare-plus' ),
					30 => esc_html__( '30 px', 'studiare-plus' ),
				],
				'default'   => 6,
				'condition' => [
					'spacing' => '1',
				],
				'devices'   => array( 'desktop', 'tablet', 'mobile' ),
				'classes'   => 'wd-hide-custom-breakpoints',
			]
		);

		$this->add_control(
			'hide_mask',
			[
				'label'        => esc_html__( 'Hide likes and comments', 'studiare-plus' ),
				'type'         => Controls_Manager::SWITCHER,
				'default'      => '0',
				'label_on'     => esc_html__( 'Yes', 'studiare-plus' ),
				'label_off'    => esc_html__( 'No', 'studiare-plus' ),
				'return_value' => '1',
			]
		);

		$this->end_controls_section();

		$this->start_controls_section(
			'images_content_section',
			[
				'label' => esc_html__( 'Images', 'studiare-plus' ),
				'tab'   => Controls_Manager::TAB_STYLE,
			]
		);

		$this->add_control(
			'aspect_ratio',
			array(
				'label'     => esc_html__( 'Aspect Ratio', 'studiare-plus' ),
				'type'      => Controls_Manager::SELECT,
				'selectors' => array(
					'{{WRAPPER}} .wd-insta' => '--wd-aspect-ratio: {{VALUE}};',
				),
				'options'   => array(
					'1/1' => '1:1',
					'4/5' => '4:5',
				),
				'default'   => '1/1',
			)
		);

		$this->add_control(
			'rounded',
			array(
				'label'        => esc_html__( 'Rounded corners for images', 'studiare-plus' ),
				'type'         => Controls_Manager::SWITCHER,
				'default'      => '0',
				'label_on'     => esc_html__( 'Yes', 'studiare-plus' ),
				'label_off'    => esc_html__( 'No', 'studiare-plus' ),
				'return_value' => '1',
			)
		);

		$this->add_control(
			'rounding_size',
			array(
				'label'     => esc_html__( 'Rounding', 'studiare-plus' ),
				'type'      => Controls_Manager::SELECT,
				'options'   => array(
					''       => esc_html__( 'Inherit', 'studiare-plus' ),
					'0'      => esc_html__( '0', 'studiare-plus' ),
					'5'      => esc_html__( '5', 'studiare-plus' ),
					'8'      => esc_html__( '8', 'studiare-plus' ),
					'12'     => esc_html__( '12', 'studiare-plus' ),
					'custom' => esc_html__( 'Custom', 'studiare-plus' ),
				),
				'default'   => '',
				'selectors' => array(
					'{{WRAPPER}}' => '--wd-brd-radius: {{VALUE}}px;',
				),
				'condition' => array(
					'rounded!' => '1',
				),
			)
		);

		$this->add_control(
			'custom_rounding_size',
			array(
				'label'      => esc_html__( 'Custom rounding', 'studiare-plus' ),
				'type'       => Controls_Manager::SLIDER,
				'size_units' => array( '%', 'px' ),
				'range'      => array(
					'px' => array(
						'min'  => 1,
						'max'  => 300,
						'step' => 1,
					),
					'%'  => array(
						'min'  => 1,
						'max'  => 100,
						'step' => 1,
					),
				),
				'selectors'  => array(
					'{{WRAPPER}}' => '--wd-brd-radius: {{SIZE}}{{UNIT}};',
				),
				'condition'  => array(
					'rounded!'      => '1',
					'rounding_size' => array( 'custom' ),
				),
			)
		);

		$this->end_controls_section();

		$this->start_controls_section(
			'content_style_section',
			array(
				'label' => esc_html__( 'Content', 'studiare-plus' ),
				'tab'   => Controls_Manager::TAB_STYLE,
			)
		);

		$this->add_responsive_control(
			'content_width',
			array(
				'label'      => esc_html__( 'Content width', 'studiare-plus' ),
				'type'       => Controls_Manager::SLIDER,
				'size_units' => array( '%', 'px' ),
				'range'      => array(
					'px' => array(
						'min'  => 1,
						'max'  => 1000,
						'step' => 1,
					),
					'%'  => array(
						'min'  => 1,
						'max'  => 100,
						'step' => 1,
					),
				),
				'selectors'  => array(
					'{{WRAPPER}} .wd-insta-cont-inner' => 'max-width: {{SIZE}}{{UNIT}};',
				),
			)
		);

		$this->add_control(
			'content_color_scheme',
			array(
				'label'   => esc_html__( 'Color Scheme', 'studiare-plus' ),
				'type'    => Controls_Manager::SELECT,
				'options' => array(
					''      => esc_html__( 'Inherit', 'studiare-plus' ),
					'light' => esc_html__( 'Light', 'studiare-plus' ),
					'dark'  => esc_html__( 'Dark', 'studiare-plus' ),
				),
				'default' => '',
			)
		);

		$this->add_control(
			'content_color',
			array(
				'label'     => esc_html__( 'Text color', 'studiare-plus' ),
				'type'      => Controls_Manager::COLOR,
				'selectors' => array(
					'{{WRAPPER}} .wd-insta-cont-inner' => 'color: {{VALUE}}',
				),
			)
		);

		$this->add_control(
			'content_bg_color',
			array(
				'label'     => esc_html__( 'Background color', 'studiare-plus' ),
				'type'      => Controls_Manager::COLOR,
				'selectors' => array(
					'{{WRAPPER}} .wd-insta-cont-inner' => 'background-color: {{VALUE}}',
				),
			)
		);

		$this->add_group_control(
			Group_Control_Typography::get_type(),
			array(
				'name'     => 'content_typography',
				'label'    => esc_html__( 'Typography', 'studiare-plus' ),
				'selector' => '{{WRAPPER}} .wd-insta-cont-inner',
			)
		);

		$this->end_controls_section();
	}

	/**
	 * Render the widget output on the frontend.
	 *
	 * Written in PHP and used to generate the final HTML.
	 *
	 * @since  1.0.0
	 *
	 * @access protected
	 */
	protected function render() {
		woodmart_elementor_instagram_template( $this->get_settings_for_display() );
	}
}

Plugin::instance()->widgets_manager->register( new Instagram() );
