<?php
/**
 * Information box map.
 */

namespace XTS\Elementor;

use Elementor\Group_Control_Background;
use Elementor\Group_Control_Border;
use Elementor\Group_Control_Image_Size;
use Elementor\Group_Control_Typography;
use Elementor\Utils;
use Elementor\Widget_Base;
use Elementor\Controls_Manager;
use Elementor\Plugin;

if ( ! defined( 'ABSPATH' ) ) {
	exit; // Direct access not allowed.
}

/**
 * Elementor widget that inserts an embeddable content into the page, from any given URL.
 *
 * @since 1.0.0
 */
class Infobox extends Widget_Base {
	/**
	 * Get widget name.
	 *
	 * @return string Widget name.
	 * @since 1.0.0
	 * @access public
	 */
	public function get_name() {
		return 'wd_infobox';
	}

	/**
	 * Get widget title.
	 *
	 * @return string Widget title.
	 * @since 1.0.0
	 * @access public
	 */
	public function get_title() {
		return esc_html__( 'Information box', 'studiare-plus' );
	}

	/**
	 * Get widget icon.
	 *
	 * @return string Widget icon.
	 * @since 1.0.0
	 * @access public
	 */
	public function get_icon() {
		return 'wd-icon-infobox';
	}

	/**
	 * Get widget categories.
	 *
	 * @return array Widget categories.
	 * @since 1.0.0
	 * @access public
	 */
	public function get_categories() {
		return [ 'wd-elements' ];
	}

	/**
	 * Register the widget controls.
	 *
	 * @since 1.0.0
	 * @access protected
	 */
	protected function register_controls() {
		/**
		 * Content tab.
		 */

		/**
		 * Icon settings.
		 */
		$this->start_controls_section(
			'icon_content_section',
			[
				'label' => esc_html__( 'Icon', 'studiare-plus' ),
			]
		);

		$this->add_control(
			'icon_type',
			[
				'label'   => esc_html__( 'Type', 'studiare-plus' ),
				'type'    => Controls_Manager::SELECT,
				'options' => [
					'icon' => esc_html__( 'Icon', 'studiare-plus' ),
					'text' => esc_html__( 'Text', 'studiare-plus' ),
				],
				'default' => 'text',
			]
		);

		$this->add_control(
			'icon_text',
			[
				'label'     => esc_html__( 'Text', 'studiare-plus' ),
				'type'      => Controls_Manager::TEXT,
				'default'   => '53',
				'condition' => [
					'icon_type' => 'text',
				],
			]
		);

		$this->add_control(
			'image',
			[
				'label'     => esc_html__( 'Choose image', 'studiare-plus' ),
				'type'      => Controls_Manager::MEDIA,
				'condition' => [
					'icon_type' => [ 'icon' ],
				],
			]
		);

		$this->add_group_control(
			Group_Control_Image_Size::get_type(),
			[
				'name'      => 'image',
				'default'   => 'thumbnail',
				'separator' => 'none',
				'condition' => [
					'icon_type' => [ 'icon' ],
				],
			]
		);

		$this->end_controls_section();

		/**
		 * Content settings.
		 */
		$this->start_controls_section(
			'content_section',
			[
				'label' => esc_html__( 'Content', 'studiare-plus' ),
			]
		);

		$this->add_control(
			'subtitle',
			[
				'label'   => esc_html__( 'Subtitle', 'studiare-plus' ),
				'type'    => Controls_Manager::TEXTAREA,
				'default' => 'Infobox subtitle text',
			]
		);

		$this->add_control(
			'title',
			[
				'label'   => esc_html__( 'Title', 'studiare-plus' ),
				'type'    => Controls_Manager::TEXTAREA,
				'default' => 'Infobox title, click to edit.',
			]
		);

		$this->add_control(
			'content',
			[
				'label'   => esc_html__( 'Content', 'studiare-plus' ),
				'type'    => Controls_Manager::WYSIWYG,
				'default' => 'Lorem ipsum, or lipsum as it is sometimes known, is dummy text used in laying out print, graphic or web designs.',
			]
		);

		$this->add_control(
			'btn_text',
			[
				'label'   => esc_html__( 'Button text', 'studiare-plus' ),
				'type'    => Controls_Manager::TEXT,
				'default' => 'Read more',
			]
		);

		$this->add_control(
			'link',
			[
				'label'   => esc_html__( 'Link', 'studiare-plus' ),
				'type'    => Controls_Manager::URL,
				'default' => [
					'url'         => '#',
					'is_external' => false,
					'nofollow'    => false,
				],
			]
		);

		$this->end_controls_section();

		/**
		 * Style tab.
		 */

		/**
		 * General settings.
		 */
		$this->start_controls_section(
			'general_style_section',
			[
				'label' => esc_html__( 'General', 'studiare-plus' ),
				'tab'   => Controls_Manager::TAB_STYLE,
			]
		);

		$this->add_control(
			'style',
			[
				'label'   => esc_html__( 'Style', 'studiare-plus' ),
				'type'    => Controls_Manager::SELECT,
				'options' => [
					'base'     => esc_html__( 'Base', 'studiare-plus' ),
					'border'   => esc_html__( 'Bordered', 'studiare-plus' ),
					'shadow'   => esc_html__( 'Shadow', 'studiare-plus' ),
					'bg-hover' => esc_html__( 'Background on hover', 'studiare-plus' ),
				],
				'default' => 'base',
			]
		);

		$this->add_control(
			'woodmart_color_scheme',
			[
				'label'   => esc_html__( 'Color Scheme', 'studiare-plus' ),
				'type'    => Controls_Manager::SELECT,
				'options' => [
					''      => esc_html__( 'Inherit', 'studiare-plus' ),
					'light' => esc_html__( 'Light', 'studiare-plus' ),
					'dark'  => esc_html__( 'Dark', 'studiare-plus' ),
				],
				'default' => '',
			]
		);

		$this->add_control(
			'alignment',
			[
				'label'   => esc_html__( 'Text alignment', 'studiare-plus' ),
				'type'    => 'wd_buttons',
				'options' => [
					'left'   => [
						'title' => esc_html__( 'Left', 'studiare-plus' ),
						'image' => WOODMART_ASSETS_IMAGES . '/settings/align/left.jpg',
					],
					'center' => [
						'title' => esc_html__( 'Center', 'studiare-plus' ),
						'image' => WOODMART_ASSETS_IMAGES . '/settings/align/center.jpg',
					],
					'right'  => [
						'title' => esc_html__( 'Right', 'studiare-plus' ),
						'image' => WOODMART_ASSETS_IMAGES . '/settings/align/right.jpg',
					],
				],
				'default' => 'center',
			]
		);

		$this->add_control(
			'image_alignment',
			[
				'label'   => esc_html__( 'Image alignment', 'studiare-plus' ),
				'type'    => 'wd_buttons',
				'options' => [
					'left'  => [
						'title' => esc_html__( 'Left', 'studiare-plus' ),
						'image' => WOODMART_ASSETS_IMAGES . '/settings/infobox/position/left.png',
					],
					'top'   => [
						'title' => esc_html__( 'Top', 'studiare-plus' ),
						'image' => WOODMART_ASSETS_IMAGES . '/settings/infobox/position/top.png',
					],
					'right' => [
						'title' => esc_html__( 'Right', 'studiare-plus' ),
						'image' => WOODMART_ASSETS_IMAGES . '/settings/infobox/position/right.png',
					],
				],
				'default' => 'top',
			]
		);

		$this->add_control(
			'image_vertical_alignment',
			array(
				'label'     => esc_html__( 'Vertical alignment', 'studiare-plus' ),
				'type'      => 'wd_buttons',
				'options'   => array(
					'top'    => array(
						'title' => esc_html__( 'Top', 'studiare-plus' ),
						'image' => WOODMART_ASSETS_IMAGES . '/settings/infobox/vertical-position/top.png',
					),
					'middle' => array(
						'title' => esc_html__( 'Middle', 'studiare-plus' ),
						'image' => WOODMART_ASSETS_IMAGES . '/settings/infobox/vertical-position/middle.png',
					),
					'bottom' => array(
						'title' => esc_html__( 'Bottom', 'studiare-plus' ),
						'image' => WOODMART_ASSETS_IMAGES . '/settings/infobox/vertical-position/bottom.png',
					),
				),
				'default'   => 'top',
				'condition' => array(
					'image_alignment' => array( 'left', 'right' ),
				),
			)
		);

		$this->add_control(
			'title_size',
			[
				'label'   => esc_html__( 'Predefined size', 'studiare-plus' ),
				'type'    => Controls_Manager::SELECT,
				'options' => [
					'default'     => esc_html__( 'Default (18px)', 'studiare-plus' ),
					'small'       => esc_html__( 'Small (16px)', 'studiare-plus' ),
					'large'       => esc_html__( 'Large (26px)', 'studiare-plus' ),
					'extra-large' => esc_html__( 'Extra Large (36px)', 'studiare-plus' ),
				],
				'default' => 'default',
			]
		);

		$this->add_responsive_control(
			'padding',
			array(
				'label'      => esc_html__( 'Padding', 'studiare-plus' ),
				'type'       => Controls_Manager::DIMENSIONS,
				'size_units' => array( 'px', '%' ),
				'selectors'  => array(
					'{{WRAPPER}} .wd-info-box' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				),
			)
		);

		$this->add_responsive_control(
			'border_radius',
			[
				'label'      => esc_html__( 'Border radius', 'studiare-plus' ),
				'type'       => Controls_Manager::DIMENSIONS,
				'size_units' => [ 'px', '%' ],
				'selectors'  => [
					'{{WRAPPER}} .wd-info-box' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				],
			]
		);

		$this->end_controls_section();

		/**
		 * Hover settings.
		 */
		$this->start_controls_section(
			'hover_style_section',
			[
				'label' => esc_html__( 'Background', 'studiare-plus' ),
				'tab'   => Controls_Manager::TAB_STYLE,
			]
		);

		$this->add_control(
			'bg_hover_colorpicker',
			[
				'label'   => esc_html__( 'Background type', 'studiare-plus' ),
				'type'    => Controls_Manager::SELECT,
				'options' => [
					'colorpicker' => esc_html__( 'Color or image', 'studiare-plus' ),
					'gradient'    => esc_html__( 'Gradient', 'studiare-plus' ),
				],
				'default' => 'colorpicker',
			]
		);

		$this->add_group_control(
			Group_Control_Background::get_type(),
			[
				'name'           => 'box_bg_color_gradient',
				'fields_options' => array(
					'background' => array(
						'label' => esc_html__( 'Background gradient',  'woodmart' ),
					),
				),
				'types'          => [ 'gradient' ],
				'selector'       => '{{WRAPPER}} .wd-info-box',
				'condition'      => [
					'bg_hover_colorpicker' => 'gradient',
				],
			]
		);

		$this->add_group_control(
			Group_Control_Background::get_type(),
			[
				'name'           => 'box_bg_hover_color_gradient',
				'fields_options' => array(
					'background' => array(
						'label' => esc_html__( 'Hover background gradient', 'studiare-plus' ),
					),
				),
				'types'          => [ 'gradient' ],
				'selector'       => '{{WRAPPER}} .wd-info-box:after',
				'condition'      => [
					'style'                => 'bg-hover',
					'bg_hover_colorpicker' => 'gradient',
				],
			]
		);

		$this->add_group_control(
			Group_Control_Background::get_type(),
			array(
				'name'           => 'box_bg_image',
				'fields_options' => array(
					'background' => array(
						'label' => esc_html__( 'Background image', 'studiare-plus' ),
					),
					'image'      => array(
						'responsive' => false,
					),
					'position'   => array(
						'label'      => esc_html__( 'Background position', 'studiare-plus' ),
						'responsive' => false,
						'options'    => [
							''              => esc_html_x( 'Default', 'Background Control', 'elementor' ),
							'center center' => esc_html_x( 'Center Center', 'Background Control', 'elementor' ),
							'center left'   => esc_html_x( 'Center Left', 'Background Control', 'elementor' ),
							'center right'  => esc_html_x( 'Center Right', 'Background Control', 'elementor' ),
							'top center'    => esc_html_x( 'Top Center', 'Background Control', 'elementor' ),
							'top left'      => esc_html_x( 'Top Left', 'Background Control', 'elementor' ),
							'top right'     => esc_html_x( 'Top Right', 'Background Control', 'elementor' ),
							'bottom center' => esc_html_x( 'Bottom Center', 'Background Control', 'elementor' ),
							'bottom left'   => esc_html_x( 'Bottom Left', 'Background Control', 'elementor' ),
							'bottom right'  => esc_html_x( 'Bottom Right', 'Background Control', 'elementor' ),
						],
					),
					'repeat'     => array(
						'label'      => esc_html__( 'Background repeat', 'studiare-plus' ),
						'responsive' => false,
					),
					'size'       => array(
						'label'      => esc_html__( 'Background size', 'studiare-plus' ),
						'responsive' => false,
						'options'    => [
							''        => esc_html_x( 'Default', 'Background Control', 'elementor' ),
							'cover'   => esc_html_x( 'Cover', 'Background Control', 'elementor' ),
							'contain' => esc_html_x( 'Contain', 'Background Control', 'elementor' ),
						],
					),
				),
				'include'        => array( 'background', 'image', 'position', 'repeat', 'size' ),
				'types'          => array( 'classic' ),
				'selector'       => '{{WRAPPER}} .wd-info-box',
				'condition'      => array(
					'bg_hover_colorpicker' => 'colorpicker',
				),
			)
		);

		$this->add_control(
			'box_bg_color',
			[
				'label'     => esc_html__( 'Background color', 'studiare-plus' ),
				'type'      => Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .wd-info-box' => 'background-color: {{VALUE}}',
				],
				'condition' => [
					'bg_hover_colorpicker' => 'colorpicker',
				],
			]
		);

		$this->add_group_control(
			Group_Control_Background::get_type(),
			array(
				'name'           => 'box_bg_hover_image',
				'fields_options' => array(
					'background' => array(
						'label' => esc_html__( 'Hover background image', 'studiare-plus' ),
					),
					'image'      => array(
						'responsive' => false,
					),
					'position'   => array(
						'label'      => esc_html__( 'Background position', 'studiare-plus' ),
						'responsive' => false,
						'options'    => [
							''              => esc_html_x( 'Default', 'Background Control', 'elementor' ),
							'center center' => esc_html_x( 'Center Center', 'Background Control', 'elementor' ),
							'center left'   => esc_html_x( 'Center Left', 'Background Control', 'elementor' ),
							'center right'  => esc_html_x( 'Center Right', 'Background Control', 'elementor' ),
							'top center'    => esc_html_x( 'Top Center', 'Background Control', 'elementor' ),
							'top left'      => esc_html_x( 'Top Left', 'Background Control', 'elementor' ),
							'top right'     => esc_html_x( 'Top Right', 'Background Control', 'elementor' ),
							'bottom center' => esc_html_x( 'Bottom Center', 'Background Control', 'elementor' ),
							'bottom left'   => esc_html_x( 'Bottom Left', 'Background Control', 'elementor' ),
							'bottom right'  => esc_html_x( 'Bottom Right', 'Background Control', 'elementor' ),
						],
					),
					'repeat'     => array(
						'label'      => esc_html__( 'Background repeat', 'studiare-plus' ),
						'responsive' => false,
					),
					'size'       => array(
						'label'      => esc_html__( 'Background size', 'studiare-plus' ),
						'responsive' => false,
						'options'    => [
							''        => esc_html_x( 'Default', 'Background Control', 'elementor' ),
							'cover'   => esc_html_x( 'Cover', 'Background Control', 'elementor' ),
							'contain' => esc_html_x( 'Contain', 'Background Control', 'elementor' ),
						],
					),
				),
				'include'        => array( 'background', 'image', 'position', 'repeat', 'size' ),
				'types'          => array( 'classic' ),
				'selector'       => '{{WRAPPER}} .wd-info-box:after',
				'condition'      => array(
					'style'                => 'bg-hover',
					'bg_hover_colorpicker' => 'colorpicker',
				),
			)
		);

		$this->add_control(
			'box_bg_hover_color',
			[
				'label'     => esc_html__( 'Hover background color', 'studiare-plus' ),
				'type'      => Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .wd-info-box:after' => 'background-color: {{VALUE}}',
				],
				'condition' => [
					'style'                => 'bg-hover',
					'bg_hover_colorpicker' => 'colorpicker',
				],
			]
		);

		$this->add_control(
			'woodmart_hover_color_scheme',
			[
				'label'   => esc_html__( 'Hover color scheme', 'studiare-plus' ),
				'type'    => Controls_Manager::SELECT,
				'options' => [
					'light' => esc_html__( 'Light', 'studiare-plus' ),
					'dark'  => esc_html__( 'Dark', 'studiare-plus' ),
				],
				'default' => 'light',
				'condition' => [
					'style' => 'bg-hover',
				],
			]
		);

		$this->end_controls_section();

		/**
		 * Icon settings.
		 */
		$this->start_controls_section(
			'icon_style_section',
			[
				'label' => esc_html__( 'Icon', 'studiare-plus' ),
				'tab'   => Controls_Manager::TAB_STYLE,
			]
		);

		$this->add_control(
			'icon_style',
			[
				'label'   => esc_html__( 'Style', 'studiare-plus' ),
				'type'    => Controls_Manager::SELECT,
				'options' => [
					'simple'      => esc_html__( 'Simple', 'studiare-plus' ),
					'with-bg'     => esc_html__( 'With background', 'studiare-plus' ),
					'with-border' => esc_html__( 'With border', 'studiare-plus' ),
				],
				'default' => 'simple',
			]
		);

		$this->add_control(
			'icon_bg_color',
			[
				'label'     => esc_html__( 'Icon background color', 'studiare-plus' ),
				'type'      => Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .wd-info-box .info-box-icon' => 'background-color: {{VALUE}}',
				],
				'condition' => [
					'icon_style' => 'with-bg',
				],
			]
		);

		$this->add_control(
			'icon_bg_hover_color',
			[
				'label'     => esc_html__( 'Icon background color on hover', 'studiare-plus' ),
				'type'      => Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .wd-info-box:hover .info-box-icon' => 'background-color: {{VALUE}}',
				],
				'condition' => [
					'icon_style' => 'with-bg',
				],
			]
		);

		$this->add_control(
			'icon_border_color',
			[
				'label'     => esc_html__( 'Icon border color', 'studiare-plus' ),
				'type'      => Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .wd-info-box .info-box-icon' => 'border-color: {{VALUE}}',
				],
				'condition' => [
					'icon_style' => 'with-border',
				],
			]
		);

		$this->add_control(
			'icon_border_hover_color',
			[
				'label'     => esc_html__( 'Icon border color on hover', 'studiare-plus' ),
				'type'      => Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .wd-info-box:hover .info-box-icon' => 'border-color: {{VALUE}}',
				],
				'condition' => [
					'icon_style' => 'with-border',
				],
			]
		);

		$this->add_control(
			'icon_text_size',
			[
				'label'     => esc_html__( 'Text size', 'studiare-plus' ),
				'type'      => Controls_Manager::SELECT,
				'options'   => [
					'default' => esc_html__( 'Default (52px)', 'studiare-plus' ),
					'small'   => esc_html__( 'Small (38px)', 'studiare-plus' ),
					'large'   => esc_html__( 'Large (74px)', 'studiare-plus' ),
				],
				'default'   => 'default',
				'condition' => [
					'icon_type' => 'text',
				],
			]
		);

		$this->add_control(
			'icon_text_color',
			[
				'label'     => esc_html__( 'Icon text color', 'studiare-plus' ),
				'type'      => Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .box-with-text' => 'color: {{VALUE}}',
				],
				'condition' => [
					'icon_type' => 'text',
				],
			]
		);

		$this->add_control(
			'icon_spacing',
			array(
				'label'     => esc_html__( 'Spacing', 'studiare-plus' ),
				'type'      => Controls_Manager::SLIDER,
				'range'     => array(
					'px' => array(
						'min'  => 5,
						'max'  => 50,
						'step' => 1,
					),
				),
				'selectors' => array(
					'{{WRAPPER}} .wd-info-box' => '--ib-icon-sp: {{SIZE}}{{UNIT}};',
				),
			)
		);

		$this->end_controls_section();

		/**
		 * Subtitle settings.
		 */
		$this->start_controls_section(
			'subtitle_style_section',
			[
				'label' => esc_html__( 'Subtitle', 'studiare-plus' ),
				'tab'   => Controls_Manager::TAB_STYLE,
			]
		);

		$this->add_control(
			'subtitle_style',
			[
				'label'   => esc_html__( 'Style', 'studiare-plus' ),
				'type'    => Controls_Manager::SELECT,
				'options' => [
					'default'    => esc_html__( 'Default', 'studiare-plus' ),
					'background' => esc_html__( 'Background', 'studiare-plus' ),
				],
				'default' => 'default',
			]
		);

		$this->add_control(
			'subtitle_custom_bg_color',
			[
				'label'     => esc_html__( 'Background color', 'studiare-plus' ),
				'type'      => Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .info-box-subtitle' => 'background-color: {{VALUE}}',
				],
				'condition' => [
					'subtitle_style' => 'background',
				],
			]
		);

		$this->add_control(
			'subtitle_color',
			[
				'label'   => esc_html__( 'Predefined color', 'studiare-plus' ),
				'type'    => Controls_Manager::SELECT,
				'options' => [
					'default' => esc_html__( 'Default', 'studiare-plus' ),
					'primary' => esc_html__( 'Primary', 'studiare-plus' ),
					'alt'     => esc_html__( 'Alternative', 'studiare-plus' ),
				],
				'default' => 'default',
			]
		);

		$this->add_control(
			'subtitle_custom_color',
			[
				'label'     => esc_html__( 'Color', 'studiare-plus' ),
				'type'      => Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .info-box-subtitle' => 'color: {{VALUE}}',
				],
			]
		);

		$this->add_group_control(
			Group_Control_Typography::get_type(),
			[
				'name'     => 'subtitle_typography',
				'label'    => esc_html__( 'Custom typography', 'studiare-plus' ),
				'selector' => '{{WRAPPER}} .info-box-subtitle',
			]
		);

		$this->end_controls_section();

		/**
		 * Title settings.
		 */
		$this->start_controls_section(
			'title_style_section',
			[
				'label' => esc_html__( 'Title', 'studiare-plus' ),
				'tab'   => Controls_Manager::TAB_STYLE,
			]
		);

		$this->add_control(
			'title_style',
			[
				'label'   => esc_html__( 'Style', 'studiare-plus' ),
				'type'    => Controls_Manager::SELECT,
				'options' => [
					'default'    => esc_html__( 'Default', 'studiare-plus' ),
					'underlined' => esc_html__( 'Underline', 'studiare-plus' ),
				],
				'default' => 'default',
			]
		);

		$this->add_control(
			'title_tag',
			[
				'label'   => esc_html__( 'Tag', 'studiare-plus' ),
				'type'    => Controls_Manager::SELECT,
				'options' => [
					'h1'   => esc_html__( 'h1', 'studiare-plus' ),
					'h2'   => esc_html__( 'h2', 'studiare-plus' ),
					'h3'   => esc_html__( 'h3', 'studiare-plus' ),
					'h4'   => esc_html__( 'h4', 'studiare-plus' ),
					'h5'   => esc_html__( 'h5', 'studiare-plus' ),
					'h6'   => esc_html__( 'h6', 'studiare-plus' ),
					'p'    => esc_html__( 'p', 'studiare-plus' ),
					'div'  => esc_html__( 'div', 'studiare-plus' ),
					'span' => esc_html__( 'span', 'studiare-plus' ),
				],
				'default' => 'h4',
			]
		);

		$this->add_control(
			'title_color',
			[
				'label'     => esc_html__( 'Color', 'studiare-plus' ),
				'type'      => Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .info-box-title' => 'color: {{VALUE}}',
				],
			]
		);

		$this->add_group_control(
			Group_Control_Typography::get_type(),
			[
				'name'     => 'title_typography',
				'label'    => esc_html__( 'Custom typography', 'studiare-plus' ),
				'selector' => '{{WRAPPER}} .info-box-title',
			]
		);

		$this->end_controls_section();

		/**
		 * Content settings.
		 */
		$this->start_controls_section(
			'content_style_section',
			[
				'label' => esc_html__( 'Content', 'studiare-plus' ),
				'tab'   => Controls_Manager::TAB_STYLE,
			]
		);

		$this->add_control(
			'custom_text_color',
			[
				'label'     => esc_html__( 'Color', 'studiare-plus' ),
				'type'      => Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .info-box-inner' => 'color: {{VALUE}}',
				],
			]
		);

		$this->add_group_control(
			Group_Control_Typography::get_type(),
			[
				'name'     => 'content_typography',
				'label'    => esc_html__( 'Custom typography', 'studiare-plus' ),
				'selector' => '{{WRAPPER}} .info-box-inner',
			]
		);

		$this->end_controls_section();

		/**
		 * Button settings.
		 */
		$this->start_controls_section(
			'button_content_section',
			[
				'label' => esc_html__( 'Button', 'studiare-plus' ),
				'tab'   => Controls_Manager::TAB_STYLE,
			]
		);

		$this->add_control(
			'btn_position',
			[
				'label'   => esc_html__( 'Button position', 'studiare-plus' ),
				'type'    => Controls_Manager::SELECT,
				'options' => [
					'hover'  => esc_html__( 'Show on hover', 'studiare-plus' ),
					'static' => esc_html__( 'Static', 'studiare-plus' ),
				],
				'default' => 'static',
			]
		);

		$this->add_control(
			'btn_size',
			[
				'label'   => esc_html__( 'Predefined size', 'studiare-plus' ),
				'type'    => Controls_Manager::SELECT,
				'options' => [
					'default'     => esc_html__( 'Default', 'studiare-plus' ),
					'extra-small' => esc_html__( 'Extra Small', 'studiare-plus' ),
					'small'       => esc_html__( 'Small', 'studiare-plus' ),
					'large'       => esc_html__( 'Large', 'studiare-plus' ),
					'extra-large' => esc_html__( 'Extra Large', 'studiare-plus' ),
				],
				'default' => 'default',
			]
		);

		$this->add_control(
			'btn_color',
			[
				'label'   => esc_html__( 'Predefined color', 'studiare-plus' ),
				'type'    => Controls_Manager::SELECT,
				'options' => [
					'default' => esc_html__( 'Default', 'studiare-plus' ),
					'primary' => esc_html__( 'Primary', 'studiare-plus' ),
					'alt'     => esc_html__( 'Alternative', 'studiare-plus' ),
					'black'   => esc_html__( 'Black', 'studiare-plus' ),
					'white'   => esc_html__( 'White', 'studiare-plus' ),
					'custom'  => esc_html__( 'Custom', 'studiare-plus' ),
				],
				'default' => 'default',
			]
		);

		$this->start_controls_tabs(
			'button_tabs_style',
			[
				'condition' => [
					'color' => [ 'custom' ],
				],
			]
		);

		$this->start_controls_tab(
			'button_tab_normal',
			[
				'label' => esc_html__( 'Normal', 'studiare-plus' ),
			]
		);

		$this->add_control(
			'bg_color',
			[
				'label'     => esc_html__( 'Background color', 'studiare-plus' ),
				'type'      => Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .wd-button-wrapper a' => 'background-color: {{VALUE}}; border-color: {{VALUE}};',
				],
			]
		);

		$this->add_control(
			'color_scheme',
			[
				'label'   => esc_html__( 'Text color scheme', 'studiare-plus' ),
				'type'    => Controls_Manager::SELECT,
				'options' => [
					'inherit' => esc_html__( 'Inherit', 'studiare-plus' ),
					'dark'    => esc_html__( 'Dark', 'studiare-plus' ),
					'light'   => esc_html__( 'Light', 'studiare-plus' ),
				],
				'default' => 'inherit',
			]
		);

		$this->end_controls_tab();

		$this->start_controls_tab(
			'button_tab_hover',
			[
				'label' => esc_html__( 'Hover', 'studiare-plus' ),
			]
		);

		$this->add_control(
			'bg_color_hover',
			[
				'label'     => esc_html__( 'Background color hover', 'studiare-plus' ),
				'type'      => Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .wd-button-wrapper:hover a' => 'background-color: {{VALUE}}; border-color: {{VALUE}};',
				],
			]
		);

		$this->add_control(
			'color_scheme_hover',
			[
				'label'   => esc_html__( 'Text color scheme on hover', 'studiare-plus' ),
				'type'    => Controls_Manager::SELECT,
				'options' => [
					'inherit' => esc_html__( 'Inherit', 'studiare-plus' ),
					'dark'    => esc_html__( 'Dark', 'studiare-plus' ),
					'light'   => esc_html__( 'Light', 'studiare-plus' ),
				],
				'default' => 'inherit',
			]
		);

		$this->end_controls_tab();

		$this->end_controls_tabs();

		$this->add_control(
			'btn_style',
			[
				'label'   => esc_html__( 'Style', 'studiare-plus' ),
				'type'    => Controls_Manager::SELECT,
				'options' => [
					'default'  => esc_html__( 'Default', 'studiare-plus' ),
					'bordered' => esc_html__( 'Bordered', 'studiare-plus' ),
					'link'     => esc_html__( 'Link button', 'studiare-plus' ),
					'3d'       => esc_html__( '3D', 'studiare-plus' ),
				],
				'default' => 'default',
			]
		);

		$this->add_control(
			'btn_shape',
			[
				'label'     => esc_html__( 'Shape', 'studiare-plus' ),
				'type'      => Controls_Manager::SELECT,
				'options'   => [
					'rectangle'  => esc_html__( 'Rectangle', 'studiare-plus' ),
					'round'      => esc_html__( 'Round', 'studiare-plus' ),
					'semi-round' => esc_html__( 'Rounded', 'studiare-plus' ),
				],
				'condition' => [
					'btn_style!' => [ 'link' ],
				],
				'default'   => 'rectangle',
			]
		);

		$this->add_control(
			'button_icon_heading',
			[
				'label'     => esc_html__( 'Icon', 'studiare-plus' ),
				'type'      => Controls_Manager::HEADING,
				'separator' => 'before',
			]
		);

		woodmart_get_button_style_icon_map( $this, 'btn_' );

		$this->add_control(
			'button_layout_heading',
			[
				'label'     => esc_html__( 'Layout', 'studiare-plus' ),
				'type'      => Controls_Manager::HEADING,
				'separator' => 'before',
			]
		);

		$this->add_control(
			'full_width',
			[
				'label'        => esc_html__( 'Full width', 'studiare-plus' ),
				'type'         => Controls_Manager::SWITCHER,
				'default'      => 'no',
				'label_on'     => esc_html__( 'Yes', 'studiare-plus' ),
				'label_off'    => esc_html__( 'No', 'studiare-plus' ),
				'return_value' => 'yes',
			]
		);

		$this->end_controls_section();

		/**
		 * Extra settings.
		 */
		$this->start_controls_section(
			'extra_content_section',
			[
				'label' => esc_html__( 'Extra', 'studiare-plus' ),
				'tab'   => Controls_Manager::TAB_STYLE,
			]
		);

		$this->add_control(
			'svg_animation',
			[
				'label'        => esc_html__( 'SVG animation', 'studiare-plus' ),
				'description'  => esc_html__( 'By default, your SVG files will not be animated.', 'studiare-plus' ),
				'type'         => Controls_Manager::SWITCHER,
				'default'      => 'no',
				'label_on'     => esc_html__( 'Yes', 'studiare-plus' ),
				'label_off'    => esc_html__( 'No', 'studiare-plus' ),
				'return_value' => 'yes',
			]
		);

		$this->end_controls_section();
	}

	/**
	 * Render the widget output on the frontend.
	 *
	 * Written in PHP and used to generate the final HTML.
	 *
	 * @since 1.0.0
	 *
	 * @access protected
	 */
	protected function render() {
		woodmart_elementor_infobox_template( $this->get_settings_for_display(), $this );
	}
}

Plugin::instance()->widgets_manager->register( new Infobox() );
