<?php
/**
 * Tabs map.
 *
 * @package studiare plus.
 */

namespace XTS\Elementor;

use Elementor\Controls_Manager;
use Elementor\Group_Control_Image_Size;
use Elementor\Group_Control_Typography;
use Elementor\Group_Control_Box_Shadow;
use Elementor\Group_Control_Border;
use Elementor\Plugin;
use Elementor\Repeater;
use Elementor\Utils;
use Elementor\Widget_Base;

if ( ! defined( 'ABSPATH' ) ) {
	exit; // Direct access not allowed.
}

/**
 * Elementor widget that adds tabs.
 */
class Tabs extends Widget_Base {
	/***
	 * This method return a widget name that will be used in the code.
	 *
	 * @return string widget name.
	 */
	public function get_name() {
		return 'wd_tabs';
	}

	/***
	 * This method return the widget title that will be displayed as the widget label.
	 *
	 * @return string widget title.
	 */
	public function get_title() {
		return esc_html__( 'Tabs', 'studiare-plus' );
	}

	/***
	 * This method set the widget icon.
	 *
	 * @return string widget icon.
	 */
	public function get_icon() {
		return 'wd-icon-tabs';
	}

	/***
	 * This method lets you set the category of the widget.
	 *
	 * @return array the category name.
	 */
	public function get_categories() {
		return array( 'wd-elements' );
	}

	/***
	 * This method lets you define which controls (setting fields) your widget will have.
	 */
	protected function register_controls() {
		/**
		 * General settings.
		 */
		$this->start_controls_section(
			'general_content_section',
			array(
				'label' => esc_html__( 'Heading', 'studiare-plus' ),
			)
		);

		$this->add_control(
			'title',
			array(
				'label'   => esc_html__( 'Title', 'studiare-plus' ),
				'type'    => Controls_Manager::TEXT,
				'default' => 'Title text example',
			)
		);

		$this->add_control(
			'description',
			[
				'label'     => esc_html__( 'Description', 'studiare-plus' ),
				'type'      => Controls_Manager::TEXTAREA,
				'condition' => [
					'design' => array( 'default', 'aside' ),
				],
			]
		);

		$this->add_control(
			'title_icon_options_separator',
			array(
				'label'     => esc_html__( 'Title icon', 'studiare-plus' ),
				'type'      => Controls_Manager::HEADING,
				'separator' => 'before',
			)
		);

		$this->add_control(
			'image',
			array(
				'label' => esc_html__( 'Choose image', 'studiare-plus' ),
				'type'  => Controls_Manager::MEDIA,
			)
		);

		$this->add_group_control(
			Group_Control_Image_Size::get_type(),
			array(
				'name'      => 'image',
				'default'   => 'thumbnail',
				'separator' => 'none',
			)
		);

		$this->end_controls_section();

		$this->start_controls_section(
			'content_section',
			array(
				'label' => esc_html__( 'Tabs', 'studiare-plus' ),
				'tab'   => Controls_Manager::TAB_CONTENT,
			)
		);

		$repeater = new Repeater();

		$repeater->start_controls_tabs( 'content_tabs' );

		$repeater->start_controls_tab(
			'content_tab',
			array(
				'label' => esc_html__( 'Content', 'studiare-plus' ),
				'tab'   => Controls_Manager::TAB_CONTENT,
			)
		);

		$repeater->add_control(
			'tab_title',
			array(
				'label'       => esc_html__( 'Title', 'studiare-plus' ),
				'placeholder' => esc_html__( 'Tab Title', 'studiare-plus' ),
				'type'        => Controls_Manager::TEXT,
				'default'     => 'Tab Title',
			)
		);

		$repeater->add_control(
			'content_type',
			array(
				'label'       => esc_html__( 'Content type', 'studiare-plus' ),
				'description' => esc_html__( 'You can display content as a simple text or if you need more complex structure you can create an HTML Block with Elementor builder and place it here.', 'studiare-plus' ),
				'type'        => Controls_Manager::SELECT,
				'options'     => array(
					'text'       => esc_html__( 'Text', 'studiare-plus' ),
					'html_block' => esc_html__( 'HTML Block', 'studiare-plus' ),
				),
				'default'     => 'text',
			)
		);

		$repeater->add_control(
			'content_text',
			array(
				'label'     => esc_html__( 'Text', 'studiare-plus' ),
				'type'      => Controls_Manager::WYSIWYG,
				'default'   => 'Lorem ipsum dolor sit amet, consectetur adipiscing elit. Ut elit tellus, luctus nec ullamcorper mattis, pulvinar dapibus leo.',
				'condition' => array(
					'content_type' => array( 'text' ),
				),
			)
		);

		$repeater->add_control(
			'content_html_block',
			array(
				'label'       => esc_html__( 'Content', 'studiare-plus' ),
				'type'        => Controls_Manager::SELECT,
				'options'     => woodmart_get_elementor_html_blocks_array(),
				'description' => function_exists( 'woodmart_get_html_block_links' ) ? woodmart_get_html_block_links() : '',
				'default'     => '0',
				'condition'   => array(
					'content_type' => array( 'html_block' ),
				),
			)
		);

		$repeater->end_controls_tab();

		$repeater->start_controls_tab(
			'icon_tab',
			array(
				'label' => esc_html__( 'Icon', 'studiare-plus' ),
				'tab'   => Controls_Manager::TAB_CONTENT,
			)
		);

		$repeater->add_control(
			'icon_type',
			array(
				'label'   => esc_html__( 'Icon type', 'studiare-plus' ),
				'type'    => Controls_Manager::SELECT,
				'options' => array(
					'icon'  => esc_html__( 'With icon', 'studiare-plus' ),
					'image' => esc_html__( 'With image', 'studiare-plus' ),
				),
				'default' => 'icon',
			)
		);

		$repeater->add_control(
			'icon',
			array(
				'label'     => esc_html__( 'Icon', 'studiare-plus' ),
				'type'      => Controls_Manager::ICONS,
				'condition' => array(
					'icon_type' => array( 'icon' ),
				),
			)
		);

		$repeater->add_control(
			'image',
			array(
				'label'     => esc_html__( 'Image', 'studiare-plus' ),
				'type'      => Controls_Manager::MEDIA,
				'default'   => array(
					'url' => Utils::get_placeholder_image_src(),
				),
				'condition' => array(
					'icon_type' => array( 'image' ),
				),
			)
		);

		$repeater->add_group_control(
			Group_Control_Image_Size::get_type(),
			array(
				'name'      => 'image',
				'separator' => 'none',
				'default'   => 'thumbnail',
				'condition' => array(
					'icon_type' => array( 'image' ),
				),
			)
		);

		$repeater->end_controls_tab();

		$repeater->end_controls_tabs();

		$this->add_control(
			'tabs',
			array(
				'label'   => esc_html__( 'Tabs items', 'studiare-plus' ),
				'type'    => Controls_Manager::REPEATER,
				'fields'  => $repeater->get_controls(),
				'default' => array(
					array(
						'tab_title' => 'Tab #1',
						'icon_type' => 'icon',
					),
					array(
						'tab_title' => 'Tab #2',
						'icon_type' => 'icon',
					),
				),
			)
		);

		$this->end_controls_section();

		/**
		 * Tabs General.
		 */
		$this->start_controls_section(
			'general_style_section',
			array(
				'label' => esc_html__( 'Heading', 'studiare-plus' ),
				'tab'   => Controls_Manager::TAB_STYLE,
			)
		);

		$this->add_control(
			'design',
			array(
				'label'   => esc_html__( 'Design', 'studiare-plus' ),
				'type'    => Controls_Manager::SELECT,
				'options' => array(
					'default' => esc_html__( 'Default', 'studiare-plus' ),
					'simple'  => esc_html__( 'Bordered', 'studiare-plus' ),
					'alt'     => esc_html__( 'Space between', 'studiare-plus' ),
					'aside'   => esc_html__( 'Aside', 'studiare-plus' ),
				),
				'default' => 'default',
			)
		);

		$this->add_control(
			'tabs_side_width',
			array(
				'label'      => esc_html__( 'Side width', 'studiare-plus' ),
				'type'       => Controls_Manager::SLIDER,
				'size_units' => array( 'px', '%' ),
				'range'      => array(
					'px' => array(
						'min'  => 100,
						'max'  => 500,
						'step' => 1,
					),
					'%'  => array(
						'min'  => 1,
						'max'  => 100,
						'step' => 1,
					),
				),
				'selectors'  => array(
					'{{WRAPPER}} .wd-tabs.tabs-design-aside' => '--wd-side-width: {{SIZE}}{{UNIT}};',
				),
				'condition'  => array(
					'design' => array( 'aside' ),
				),
			)
		);

		$this->add_responsive_control(
			'space_between_tabs_title_vertical',
			array(
				'label'     => esc_html__( 'Spacing', 'studiare-plus' ),
				'type'      => Controls_Manager::SLIDER,
				'range'     => array(
					'px' => array(
						'min'  => 0,
						'max'  => 150,
						'step' => 1,
					),
				),
				'selectors' => array(
					'{{WRAPPER}} .wd-tabs' => '--wd-row-gap: {{SIZE}}px;',
				),
				'devices'   => array( 'desktop', 'tablet', 'mobile' ),
				'classes'   => 'wd-hide-custom-breakpoints',
			)
		);

		$this->add_control(
			'enable_heading_bg',
			array(
				'label'        => esc_html__( 'Background color', 'studiare-plus' ),
				'type'         => Controls_Manager::SWITCHER,
				'default'      => 'no',
				'label_on'     => esc_html__( 'Yes', 'studiare-plus' ),
				'label_off'    => esc_html__( 'No', 'studiare-plus' ),
				'return_value' => 'yes',
			)
		);

		$this->add_control(
			'heading_bg',
			array(
				'label'     => esc_html__( 'Background color', 'studiare-plus' ),
				'type'      => Controls_Manager::COLOR,
				'selectors' => array(
					'{{WRAPPER}} .wd-tabs.wd-header-with-bg .wd-tabs-header' => 'background-color: {{VALUE}}',
				),
				'condition' => array(
					'enable_heading_bg' => 'yes',
				),
			)
		);

		$this->add_control(
			'tabs_alignment',
			array(
				'label'     => esc_html__( 'Alignment', 'studiare-plus' ),
				'type'      => 'wd_buttons',
				'options'   => array(
					'left'   => array(
						'title' => esc_html__( 'Left', 'studiare-plus' ),
						'image' => WOODMART_ASSETS_IMAGES . '/settings/align/left.jpg',
					),
					'center' => array(
						'title' => esc_html__( 'Center', 'studiare-plus' ),
						'image' => WOODMART_ASSETS_IMAGES . '/settings/align/center.jpg',
					),
					'right'  => array(
						'title' => esc_html__( 'Right', 'studiare-plus' ),
						'image' => WOODMART_ASSETS_IMAGES . '/settings/align/right.jpg',
					),
				),
				'default'   => 'center',
				'condition' => array(
					'design' => array( 'default' ),
				),
			)
		);

		$this->add_control(
			'title_options_separator',
			array(
				'label'     => esc_html__( 'Title', 'studiare-plus' ),
				'type'      => Controls_Manager::HEADING,
				'separator' => 'before',
			)
		);

		$this->add_group_control(
			Group_Control_Typography::get_type(),
			array(
				'name'     => 'tabs_heading_typography',
				'label'    => esc_html__( 'Typography', 'studiare-plus' ),
				'selector' => '{{WRAPPER}} .wd-tabs .tabs-name',
			)
		);

		$this->add_control(
			'tabs_title_color',
			array(
				'label'     => esc_html__( 'Color', 'studiare-plus' ),
				'type'      => Controls_Manager::COLOR,
				'selectors' => array(
					'{{WRAPPER}} .tabs-name' => 'color: {{VALUE}}',
				),
			)
		);

		$this->add_control(
			'tabs_border_color',
			array(
				'label'     => esc_html__( 'Border color', 'studiare-plus' ),
				'type'      => Controls_Manager::COLOR,
				'selectors' => array(
					'{{WRAPPER}} .wd-tabs.tabs-design-simple .tabs-name' => 'border-color: {{VALUE}}',
				),
				'condition' => array(
					'design' => array( 'simple' ),
				),
			)
		);

		$this->add_control(
			'description_options_separator',
			array(
				'label'     => esc_html__( 'Description', 'studiare-plus' ),
				'type'      => Controls_Manager::HEADING,
				'separator' => 'before',
				'condition' => array(
					'design' => array( 'default', 'aside' ),
				),
			)
		);

		$this->add_control(
			'tabs_description_color',
			array(
				'label'     => esc_html__( 'Color', 'studiare-plus' ),
				'type'      => Controls_Manager::COLOR,
				'selectors' => array(
					'{{WRAPPER}} .wd-tabs-desc' => 'color: {{VALUE}}',
				),
				'condition' => array(
					'design' => array( 'default', 'aside' ),
				),
			)
		);

		$this->end_controls_section();
		/**
		 * Tabs Style.
		 */
		$this->start_controls_section(
			'title_style_section',
			array(
				'label' => esc_html__( 'Navigation', 'studiare-plus' ),
				'tab'   => Controls_Manager::TAB_STYLE,
			)
		);

		$this->add_control(
			'tabs_style',
			array(
				'label'   => esc_html__( 'Style', 'studiare-plus' ),
				'type'    => Controls_Manager::SELECT,
				'options' => array(
					'default'   => esc_html__( 'Default', 'studiare-plus' ),
					'underline' => esc_html__( 'Underline', 'studiare-plus' ),
				),
				'default' => 'default',
			)
		);

		$this->add_group_control(
			Group_Control_Typography::get_type(),
			array(
				'name'     => 'tabs_title_typography',
				'label'    => esc_html__( 'Typography', 'studiare-plus' ),
				'selector' => '{{WRAPPER}} .wd-nav-tabs > li > a',
			)
		);

		$this->start_controls_tabs( 'tabs_title_text_color_tabs' );

		$this->start_controls_tab(
			'tabs_title_text_color_tab',
			array(
				'label' => esc_html__( 'Idle', 'studiare-plus' ),
			)
		);

		$this->add_control(
			'tabs_title_text_idle_color',
			array(
				'label'     => esc_html__( 'Color', 'studiare-plus' ),
				'type'      => Controls_Manager::COLOR,
				'selectors' => array(
					'{{WRAPPER}} .wd-nav-tabs' => '--nav-color: {{VALUE}}',
				),
			)
		);

		$this->add_control(
			'tabs_title_bg_color_enable',
			array(
				'label'        => esc_html__( 'Background color', 'studiare-plus' ),
				'type'         => Controls_Manager::SWITCHER,
				'default'      => '0',
				'label_on'     => esc_html__( 'Yes', 'studiare-plus' ),
				'label_off'    => esc_html__( 'No', 'studiare-plus' ),
				'return_value' => '1',
				'render_type'  => 'template',
			)
		);

		$this->add_control(
			'tabs_title_bg_color_idle',
			array(
				'label'     => esc_html__( 'Background color', 'studiare-plus' ),
				'type'      => Controls_Manager::COLOR,
				'selectors' => array(
					'{{WRAPPER}} .wd-nav-tabs' => '--nav-bg: {{VALUE}}',
				),
				'condition' => array(
					'tabs_title_bg_color_enable' => '1',
				),
			)
		);

		$this->add_control(
			'tabs_title_border_enable',
			array(
				'label'        => esc_html__( 'Border', 'studiare-plus' ),
				'type'         => Controls_Manager::SWITCHER,
				'default'      => '0',
				'label_on'     => esc_html__( 'Yes', 'studiare-plus' ),
				'label_off'    => esc_html__( 'No', 'studiare-plus' ),
				'return_value' => '1',
				'render_type'  => 'template',
			)
		);

		$this->add_group_control(
			Group_Control_Border::get_type(),
			array(
				'name'      => 'tabs_title_border',
				'selector'  => '{{WRAPPER}} .wd-nav-tabs > li > a',
				'condition' => array(
					'tabs_title_border_enable' => '1',
				),
			)
		);

		$this->add_responsive_control(
			'tabs_title_border_radius',
			array(
				'label'      => esc_html__( 'Border Radius', 'studiare-plus' ),
				'type'       => Controls_Manager::DIMENSIONS,
				'size_units' => array( 'px', '%', 'em' ),
				'selectors'  => array(
					'{{WRAPPER}} .wd-nav-tabs > li > a' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				),
				'condition'  => array(
					'tabs_title_border_enable' => '1',
				),
			)
		);

		$this->add_control(
			'tabs_title_box_shadow_enable',
			array(
				'label'        => esc_html__( 'Box shadow', 'studiare-plus' ),
				'type'         => Controls_Manager::SWITCHER,
				'default'      => '0',
				'label_on'     => esc_html__( 'Yes', 'studiare-plus' ),
				'label_off'    => esc_html__( 'No', 'studiare-plus' ),
				'return_value' => '1',
				'render_type'  => 'template',
			)
		);

		$this->add_group_control(
			Group_Control_Box_Shadow::get_type(),
			array(
				'name'      => 'tabs_box_shadow',
				'selector'  => '{{WRAPPER}} .wd-nav-tabs > li > a',
				'condition' => array(
					'tabs_title_box_shadow_enable' => '1',
				),
			)
		);

		$this->end_controls_tab();

		$this->start_controls_tab(
			'tabs_title_text_hover_color_tab',
			array(
				'label' => esc_html__( 'Hover', 'studiare-plus' ),
			)
		);

		$this->add_control(
			'tabs_title_text_hover_color',
			array(
				'label'     => esc_html__( 'Color', 'studiare-plus' ),
				'type'      => Controls_Manager::COLOR,
				'selectors' => array(
					'{{WRAPPER}} .wd-nav-tabs' => '--nav-color-hover: {{VALUE}}',
				),
			)
		);

		$this->add_control(
			'tabs_title_bg_hover_color_enable',
			array(
				'label'        => esc_html__( 'Background color', 'studiare-plus' ),
				'type'         => Controls_Manager::SWITCHER,
				'default'      => '0',
				'label_on'     => esc_html__( 'Yes', 'studiare-plus' ),
				'label_off'    => esc_html__( 'No', 'studiare-plus' ),
				'return_value' => '1',
				'render_type'  => 'template',
			)
		);

		$this->add_control(
			'tabs_title_bg_color_hover',
			array(
				'label'     => esc_html__( 'Background color', 'studiare-plus' ),
				'type'      => Controls_Manager::COLOR,
				'selectors' => array(
					'{{WRAPPER}} .wd-nav-tabs' => '--nav-bg-hover: {{VALUE}}',
				),
				'condition' => array(
					'tabs_title_bg_hover_color_enable' => '1',
				),
			)
		);

		$this->add_control(
			'tabs_title_border_hover_enable',
			array(
				'label'        => esc_html__( 'Border', 'studiare-plus' ),
				'type'         => Controls_Manager::SWITCHER,
				'default'      => '0',
				'label_on'     => esc_html__( 'Yes', 'studiare-plus' ),
				'label_off'    => esc_html__( 'No', 'studiare-plus' ),
				'return_value' => '1',
				'render_type'  => 'template',
			)
		);

		$this->add_group_control(
			Group_Control_Border::get_type(),
			array(
				'name'      => 'tabs_border_hover',
				'selector'  => '{{WRAPPER}} .wd-nav-tabs > li:hover > a',
				'condition' => array(
					'tabs_title_border_hover_enable' => '1',
				),
			)
		);

		$this->add_control(
			'tabs_title_box_shadow_hover_enable',
			array(
				'label'        => esc_html__( 'Box shadow', 'studiare-plus' ),
				'type'         => Controls_Manager::SWITCHER,
				'default'      => '0',
				'label_on'     => esc_html__( 'Yes', 'studiare-plus' ),
				'label_off'    => esc_html__( 'No', 'studiare-plus' ),
				'return_value' => '1',
				'render_type'  => 'template',
			)
		);

		$this->add_group_control(
			Group_Control_Box_Shadow::get_type(),
			array(
				'name'      => 'tabs_box_shadow_hover',
				'selector'  => '{{WRAPPER}} .wd-nav-tabs > li:hover > a',
				'condition' => array(
					'tabs_title_box_shadow_hover_enable' => '1',
				),
			)
		);

		$this->end_controls_tab();

		$this->start_controls_tab(
			'tabs_title_text_hover_active_tab',
			array(
				'label' => esc_html__( 'Active', 'studiare-plus' ),
			)
		);

		$this->add_control(
			'tabs_title_text_hover_active',
			array(
				'label'     => esc_html__( 'Color', 'studiare-plus' ),
				'type'      => Controls_Manager::COLOR,
				'selectors' => array(
					'{{WRAPPER}} .wd-nav-tabs' => '--nav-color-active: {{VALUE}}',
				),
			)
		);

		$this->add_control(
			'tabs_title_bg_active_color_enable',
			array(
				'label'        => esc_html__( 'Background color', 'studiare-plus' ),
				'type'         => Controls_Manager::SWITCHER,
				'default'      => '0',
				'label_on'     => esc_html__( 'Yes', 'studiare-plus' ),
				'label_off'    => esc_html__( 'No', 'studiare-plus' ),
				'return_value' => '1',
				'render_type'  => 'template',
			)
		);

		$this->add_control(
			'tabs_bg_color_active',
			array(
				'label'     => esc_html__( 'Background color', 'studiare-plus' ),
				'type'      => Controls_Manager::COLOR,
				'selectors' => array(
					'{{WRAPPER}} .wd-nav-tabs' => '--nav-bg-active: {{VALUE}}',
				),
				'condition' => array(
					'tabs_title_bg_active_color_enable' => '1',
				),
			)
		);

		$this->add_control(
			'tabs_title_border_active_enable',
			array(
				'label'        => esc_html__( 'Border', 'studiare-plus' ),
				'type'         => Controls_Manager::SWITCHER,
				'default'      => '0',
				'label_on'     => esc_html__( 'Yes', 'studiare-plus' ),
				'label_off'    => esc_html__( 'No', 'studiare-plus' ),
				'return_value' => '1',
				'render_type'  => 'template',
			)
		);

		$this->add_group_control(
			Group_Control_Border::get_type(),
			array(
				'name'      => 'tabs_border_active',
				'selector'  => '{{WRAPPER}} :is(.wd-nav-tabs > li.wd-active > a, .wd-tabs:not(.wd-inited) .wd-nav-tabs > li:first-child > a)',
				'condition' => array(
					'tabs_title_border_active_enable' => '1',
				),
			)
		);

		$this->add_control(
			'tabs_title_box_shadow_active_enable',
			array(
				'label'        => esc_html__( 'Box shadow', 'studiare-plus' ),
				'type'         => Controls_Manager::SWITCHER,
				'default'      => '0',
				'label_on'     => esc_html__( 'Yes', 'studiare-plus' ),
				'label_off'    => esc_html__( 'No', 'studiare-plus' ),
				'return_value' => '1',
				'render_type'  => 'template',
			)
		);

		$this->add_group_control(
			Group_Control_Box_Shadow::get_type(),
			array(
				'name'      => 'tabs_box_shadow_active',
				'selector'  => '{{WRAPPER}} :is(.wd-nav-tabs > li.wd-active > a, .wd-tabs:not(.wd-inited) .wd-nav-tabs > li:first-child > a)',
				'condition' => array(
					'tabs_title_box_shadow_active_enable' => '1',
				),
			)
		);

		$this->end_controls_tab();

		$this->end_controls_tabs();

		$this->add_control(
			'items_separator',
			array(
				'type' => Controls_Manager::DIVIDER,
			)
		);

		$this->add_responsive_control(
			'space_between_tabs_title_horizontal',
			array(
				'label'     => esc_html__( 'Items gap', 'studiare-plus' ),
				'type'      => Controls_Manager::SLIDER,
				'range'     => array(
					'px' => array(
						'min'  => 0,
						'max'  => 150,
						'step' => 1,
					),
				),
				'selectors' => array(
					'{{WRAPPER}} .wd-nav-tabs' => '--nav-gap: {{SIZE}}px;',
				),
			)
		);

		$this->add_responsive_control(
			'tabs_title_padding',
			array(
				'label'      => esc_html__( 'Padding', 'studiare-plus' ),
				'type'       => Controls_Manager::DIMENSIONS,
				'size_units' => array( 'px', '%', 'em' ),
				'selectors'  => array(
					'{{WRAPPER}} .wd-nav-tabs' => '--nav-pd: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				),
			)
		);

		$this->add_control(
			'title_text_color_scheme',
			array(
				'label'   => esc_html__( 'Color scheme', 'studiare-plus' ),
				'type'    => Controls_Manager::SELECT,
				'options' => array(
					'inherit' => esc_html__( 'Inherit', 'studiare-plus' ),
					'light'   => esc_html__( 'Light', 'studiare-plus' ),
					'dark'    => esc_html__( 'Dark', 'studiare-plus' ),
				),
				'default' => 'inherit',
			)
		);

		$this->add_control(
			'icon_options_separator',
			array(
				'label'     => esc_html__( 'Icon', 'studiare-plus' ),
				'type'      => Controls_Manager::HEADING,
				'separator' => 'before',
			)
		);

		$this->add_control(
			'icon_alignment',
			array(
				'label'   => esc_html__( 'Alignment', 'studiare-plus' ),
				'type'    => 'wd_buttons',
				'options' => array(
					'left'  => array(
						'title' => esc_html__( 'Left', 'studiare-plus' ),
						'image' => WOODMART_ASSETS_IMAGES . '/settings/infobox/position/left.png',
					),
					'top'   => array(
						'title' => esc_html__( 'Top', 'studiare-plus' ),
						'image' => WOODMART_ASSETS_IMAGES . '/settings/infobox/position/top.png',
					),
					'right' => array(
						'title' => esc_html__( 'Right', 'studiare-plus' ),
						'image' => WOODMART_ASSETS_IMAGES . '/settings/infobox/position/right.png',
					),
				),
				'default' => 'left',
			)
		);

		$this->end_controls_section();

		$this->start_controls_section(
			'content_style_section',
			array(
				'label' => esc_html__( 'Content', 'studiare-plus' ),
				'tab'   => Controls_Manager::TAB_STYLE,
			)
		);

		$this->add_group_control(
			Group_Control_Typography::get_type(),
			array(
				'name'     => 'tabs_content_typography',
				'label'    => esc_html__( 'Typography', 'studiare-plus' ),
				'selector' => '{{WRAPPER}} .wd-tab-content',
			)
		);

		$this->add_control(
			'content_text_color_scheme',
			array(
				'label'   => esc_html__( 'Color scheme', 'studiare-plus' ),
				'type'    => Controls_Manager::SELECT,
				'options' => array(
					'inherit' => esc_html__( 'Inherit', 'studiare-plus' ),
					'light'   => esc_html__( 'Light', 'studiare-plus' ),
					'dark'    => esc_html__( 'Dark', 'studiare-plus' ),
					'custom'  => esc_html__( 'Custom', 'studiare-plus' ),
				),
				'default' => 'inherit',
			)
		);

		$this->add_control(
			'custom_content_text_color_scheme',
			array(
				'label'     => esc_html__( 'Custom color', 'studiare-plus' ),
				'type'      => Controls_Manager::COLOR,
				'selectors' => array(
					'{{WRAPPER}} .wd-tab-content' => 'color: {{VALUE}}',
				),
				'condition' => array(
					'content_text_color_scheme' => 'custom',
				),
			)
		);

		$this->end_controls_section();
	}

	/***
	 * This method render the code and generate the final HTML on the frontend using PHP.
	 */
	protected function render() {
		$tabs_parent_settings = $this->get_settings_for_display();
		$tabs_child_settings  = $tabs_parent_settings['tabs'];

		$tabs_title_bg_activated      = $tabs_parent_settings['tabs_title_bg_color_enable'] || $tabs_parent_settings['tabs_title_bg_hover_color_enable'] || $tabs_parent_settings['tabs_title_bg_active_color_enable'];
		$tabs_title_box_shadow_active = $tabs_parent_settings['tabs_title_box_shadow_enable'] || $tabs_parent_settings['tabs_title_box_shadow_hover_enable'] || $tabs_parent_settings['tabs_title_box_shadow_active_enable'];
		$tabs_title_border_active     = $tabs_parent_settings['tabs_title_border_enable'] || $tabs_parent_settings['tabs_title_border_hover_enable'] || $tabs_parent_settings['tabs_title_border_active_enable'];

		$wrapper_classes         = ' tabs-design-' . $tabs_parent_settings['design'];
		$nav_tabs_title_classes  = ' wd-icon-pos-' . $tabs_parent_settings['icon_alignment'];
		$nav_tabs_title_classes .= ' wd-style-' . $tabs_parent_settings['tabs_style'];

		if ( $tabs_title_bg_activated || $tabs_title_box_shadow_active || $tabs_title_border_active ) {
			$nav_tabs_title_classes .= ' wd-add-pd';
		}

		$this->add_inline_editing_attributes( 'content_text' );

		woodmart_enqueue_js_script( 'tabs-element' );
		woodmart_enqueue_inline_style( 'tabs' );

		if ( ! $tabs_parent_settings['tabs_alignment'] ) {
			$tabs_parent_settings['tabs_alignment'] = 'center';
		}

		if ( 'yes' === $tabs_parent_settings['enable_heading_bg'] ) {
			$wrapper_classes .= ' wd-header-with-bg';
		}

		$nav_tabs_wrapper_classes = '';

		if ( 'default' === $tabs_parent_settings['design'] ) {
			$nav_tabs_wrapper_classes .= ' text-' . $tabs_parent_settings['tabs_alignment'];
		}

		$image_output = '';

		// Image settings.
		$custom_image_size = isset( $tabs_parent_settings['image_custom_dimension']['width'] ) && $tabs_parent_settings['image_custom_dimension']['width'] ? $tabs_parent_settings['image_custom_dimension'] : array(
			'width'  => 128,
			'height' => 128,
		);

		if ( isset( $tabs_parent_settings['image']['id'] ) && $tabs_parent_settings['image']['id'] ) {
			$image_output              = '<span class="img-wrapper">' . woodmart_otf_get_image_html( $tabs_parent_settings['image']['id'], $tabs_parent_settings['image_size'], $tabs_parent_settings['image_custom_dimension'] ) . '</span>';
			$render_svg_with_image_tag = apply_filters( 'woodmart_render_svg_with_image_tag', true );

			if ( woodmart_is_svg( $tabs_parent_settings['image']['url'] ) ) {
				if ( $render_svg_with_image_tag ) {
					$custom_image_size = 'custom' !== $tabs_parent_settings['image_size'] && 'full' !== $tabs_parent_settings['image_size'] ? $tabs_parent_settings['image_size'] : $custom_image_size;
					$image_output      = '<span class="img-wrapper">' .  woodmart_get_svg_html( $tabs_parent_settings['image']['id'], $custom_image_size ) . '</span>';
				} else {
					$image_output = '<span class="svg-icon img-wrapper" style="width:' . esc_attr( $custom_image_size['width'] ) . 'px; height:' . esc_attr( $custom_image_size['height'] ) . 'px;">' . woodmart_get_any_svg( $tabs_parent_settings['image']['url'], wp_rand( 999, 9999 ) ) . '</span>';
				}
			}
		}

		$tabs_nav_wrapper_classes = '';
		if ( 'inherit' !== $tabs_parent_settings['title_text_color_scheme'] ) {
			$tabs_nav_wrapper_classes .= ' color-scheme-' . $tabs_parent_settings['title_text_color_scheme'];
		}

		$content_classes_wrapper = '';
		if ( 'inherit' !== $tabs_parent_settings['content_text_color_scheme'] && 'custom' !== $tabs_parent_settings['content_text_color_scheme'] ) {
			$content_classes_wrapper = ' color-scheme-' . $tabs_parent_settings['content_text_color_scheme'];
		}
		?>
		<div class="wd-tabs<?php echo esc_attr( $wrapper_classes ); ?>">
			<div class="wd-tabs-header<?php echo esc_attr( $nav_tabs_wrapper_classes ); ?>">
				<?php if ( $tabs_parent_settings['title'] ) : ?>
					<div class="tabs-name title">
						<?php if ( $image_output ) : ?>
							<?php echo $image_output; // phpcs:ignore ?>
						<?php endif; ?>

						<span class="tabs-text" data-elementor-setting-key="title">
							<?php echo wp_kses( $tabs_parent_settings['title'], woodmart_get_allowed_html() ); ?>
						</span>
					</div>
				<?php endif; ?>

				<?php if ( $tabs_parent_settings['description'] ) : ?>
					<div class="wd-tabs-desc">
						<?php echo wp_kses( $tabs_parent_settings['description'], woodmart_get_allowed_html() ); ?>
					</div>
				<?php endif; ?>

				<div class="wd-nav-wrapper wd-nav-tabs-wrapper<?php echo esc_attr( $tabs_nav_wrapper_classes ); ?>">
					<ul class="wd-nav wd-nav-tabs<?php echo esc_attr( $nav_tabs_title_classes ); ?>">
						<?php foreach ( $tabs_child_settings as $id => $tab ) : ?>
							<?php
							// Icon settings.

							$image_size = array(
								'width'  => 128,
								'height' => 128,
							);

							if ( isset( $tab['image_size'] ) && ! empty( $tab['image_size'] ) && 'custom' !== $tab['image_size'] ) {
								$image_size = $tab['image_size'];
							} elseif ( 'custom' === $tab['image_size'] && isset( $tab['image_custom_dimension']['width'] ) && ! empty( $tab['image_custom_dimension']['width'] ) ) {
								$image_size = $tab['image_custom_dimension'];
							}

							$icon_output = '';

							if ( 'image' === $tab['icon_type'] && isset( $tab['image']['id'] ) && $tab['image']['id'] ) {
								$icon_output = woodmart_otf_get_image_html( $tab['image']['id'], $tab['image_size'], $tab['image_custom_dimension'] );

								if ( woodmart_is_svg( $tab['image']['url'] ) ) {
									$icon_output = woodmart_get_svg_html( $tab['image']['id'], $image_size, array( 'class' => 'svg-icon' ) );
								}
							} elseif ( 'icon' === $tab['icon_type'] && $tab['icon'] ) {
								$icon_output = woodmart_elementor_get_render_icon( $tab['icon'] );
							}
							?>

							<li>
								<a href="#" class="wd-nav-link">
									<?php if ( ! empty( $icon_output ) ) : ?>
										<span class="img-wrapper">
											<?php echo $icon_output; // phpcs:ignore ?>
										</span>
									<?php endif; ?>

									<span class="nav-link-text wd-tabs-title">
										<?php echo esc_html( $tab['tab_title'] ); ?>
									</span>
								</a>
							</li>

						<?php endforeach; ?>
					</ul>
				</div>
			</div>

			<div class="wd-tabs-content-wrapper<?php echo esc_attr( $content_classes_wrapper ); ?>">
				<?php foreach ( $tabs_child_settings as $index => $tab ) : ?>
					<?php $tab_content_setting_key = $this->get_repeater_setting_key( 'content_text', 'tabs', $index ); ?>
					<?php $this->add_inline_editing_attributes( $tab_content_setting_key ); ?>

					<div class="wd-tab-content wd-entry-content">
						<?php if ( 'html_block' === $tab['content_type'] ) : ?>
							<?php echo woodmart_get_html_block( $tab['content_html_block'] ); // phpcs:ignore ?>
						<?php elseif ( 'text' === $tab['content_type'] ) : ?>
							<div <?php echo $this->get_render_attribute_string( $tab_content_setting_key ); ?>>
								<?php echo do_shortcode( $tab['content_text'] ); ?>
							</div>
						<?php endif; ?>
					</div>

				<?php endforeach; ?>
			</div>
		</div>
		<?php
	}
}

Plugin::instance()->widgets_manager->register( new Tabs() );
