<?php
/**
 * Counter map.
 */

namespace XTS\Elementor;

use Elementor\Group_Control_Typography;
use Elementor\Widget_Base;
use Elementor\Controls_Manager;
use Elementor\Plugin;

if ( ! defined( 'ABSPATH' ) ) {
	exit; // Direct access not allowed.
}

/**
 * Elementor widget that inserts an embeddable content into the page, from any given URL.
 *
 * @since 1.0.0
 */
class Counter extends Widget_Base {
	/**
	 * Get widget name.
	 *
	 * @since 1.0.0
	 * @access public
	 *
	 * @return string Widget name.
	 */
	public function get_name() {
		return 'wd_counter';
	}

	/**
	 * Get widget title.
	 *
	 * @since 1.0.0
	 * @access public
	 *
	 * @return string Widget title.
	 */
	public function get_title() {
		return esc_html__( 'Animated Counter', 'studiare-plus' );
	}

	/**
	 * Get widget icon.
	 *
	 * @since 1.0.0
	 * @access public
	 *
	 * @return string Widget icon.
	 */
	public function get_icon() {
		return 'wd-icon-counter';
	}

	/**
	 * Get widget categories.
	 *
	 * @since 1.0.0
	 * @access public
	 *
	 * @return array Widget categories.
	 */
	public function get_categories() {
		return [ 'wd-elements' ];
	}

	/**
	 * Register the widget controls.
	 *
	 * @since 1.0.0
	 * @access protected
	 */
	protected function register_controls() {
		/**
		 * Content tab.
		 */

		/**
		 * General settings.
		 */
		$this->start_controls_section(
			'general_content_section',
			[
				'label' => esc_html__( 'General', 'studiare-plus' ),
			]
		);

		$this->add_control(
			'label',
			[
				'label'   => esc_html__( 'Label', 'studiare-plus' ),
				'type'    => Controls_Manager::TEXT,
				'default' => 'Projects',
			]
		);

		$this->add_control(
			'start_value',
			[
				'label'   => esc_html__( 'Start value', 'studiare-plus' ),
				'type'    => Controls_Manager::NUMBER,
				'default' => '',
			]
		);

		$this->add_control(
			'value',
			[
				'label'       => esc_html__( 'Actual value', 'studiare-plus' ),
				'description' => esc_html__( 'Our final point. For ex.: 95', 'studiare-plus' ),
				'type'        => Controls_Manager::NUMBER,
				'default'     => '45',
			]
		);

		$this->end_controls_section();

		/**
		 * Style tab.
		 */

		/**
		 * General settings.
		 */
		$this->start_controls_section(
			'general_style_section',
			[
				'label' => esc_html__( 'General', 'studiare-plus' ),
				'tab'   => Controls_Manager::TAB_STYLE,
			]
		);

		$this->add_control(
			'size',
			[
				'label'   => esc_html__( 'Size', 'studiare-plus' ),
				'type'    => Controls_Manager::SELECT,
				'options' => [
					''            => esc_html__( 'Default', 'studiare-plus' ),
					'small'       => esc_html__( 'Small', 'studiare-plus' ),
					'large'       => esc_html__( 'Large', 'studiare-plus' ),
					'extra-large' => esc_html__( 'Extra large', 'studiare-plus' ),
				],
				'default' => '',
			]
		);

		$this->add_group_control(
			Group_Control_Typography::get_type(),
			[
				'name'     => 'value_typography',
				'label'    => esc_html__( 'Value typography', 'studiare-plus' ),
				'selector' => '{{WRAPPER}} .counter-value',
			]
		);

		$this->add_control(
			'align',
			[
				'label'   => esc_html__( 'Align', 'studiare-plus' ),
				'type'    => 'wd_buttons',
				'options' => [
					'left'   => [
						'title' => esc_html__( 'Left', 'studiare-plus' ),
						'image' => WOODMART_ASSETS_IMAGES . '/settings/align/left.jpg',
					],
					'center' => [
						'title' => esc_html__( 'Center', 'studiare-plus' ),
						'image' => WOODMART_ASSETS_IMAGES . '/settings/align/center.jpg',
					],
					'right'  => [
						'title' => esc_html__( 'Right', 'studiare-plus' ),
						'image' => WOODMART_ASSETS_IMAGES . '/settings/align/right.jpg',
					],
				],
				'default' => 'center',
			]
		);

		$this->add_control(
			'color_scheme',
			[
				'label'   => esc_html__( 'Color scheme', 'studiare-plus' ),
				'type'    => Controls_Manager::SELECT,
				'options' => [
					''       => esc_html__( 'Inherit', 'studiare-plus' ),
					'light'  => esc_html__( 'Light', 'studiare-plus' ),
					'dark'   => esc_html__( 'Dark', 'studiare-plus' ),
					'custom' => esc_html__( 'Custom', 'studiare-plus' ),
				],
				'default' => '',
			]
		);

		$this->add_control(
			'color',
			[
				'label'     => esc_html__( 'Custom color', 'studiare-plus' ),
				'type'      => Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .woodmart-counter' => 'color: {{VALUE}}',
				],
				'condition' => [
					'color_scheme' => 'custom',
				],
			]
		);

		$this->end_controls_section();
	}

	/**
	 * Render the widget output on the frontend.
	 *
	 * Written in PHP and used to generate the final HTML.
	 *
	 * @since 1.0.0
	 *
	 * @access protected
	 */
	protected function render() {
		$default_settings = [
			'label'        => '',
			'value'        => 100,
			'start_value'  => 0,

			'color_scheme' => '',
			'size'         => 'default',
			'align'        => 'center',
		];

		$settings = wp_parse_args( $this->get_settings_for_display(), $default_settings );

		$this->add_render_attribute(
			[
				'wrapper' => [
					'class' => [
						'woodmart-counter',
						'counter-' . $settings['size'],
						'text-' . $settings['align'],
						'color-scheme-' . $settings['color_scheme'],
					],
				],
				'counter' => [
					'class'      => [
						'counter-value',
					],
					'data-state' => [
						'new',
					],
					'data-final' => [
						$settings['value'],
					],
				],
				'label'   => [
					'class' => [
						'counter-label',
					],
				],
			]
		);

		$this->add_inline_editing_attributes( 'label' );

		woodmart_enqueue_js_library( 'waypoints' );
		woodmart_enqueue_js_script( 'counter-element' );
		woodmart_enqueue_inline_style( 'counter' );

		?>
		<div <?php echo $this->get_render_attribute_string( 'wrapper' ); ?>>
			<div <?php echo $this->get_render_attribute_string( 'counter' ); ?>>
				<?php echo ! empty( $settings['start_value'] ) ? esc_html( $settings['start_value'] ) : 0; ?>
			</div>

			<?php if ( $settings['label'] ) : ?>
				<div <?php echo $this->get_render_attribute_string( 'label' ); ?>>
					<?php echo esc_html( $settings['label'] ); ?>
				</div>
			<?php endif ?>
		</div>
		<?php
	}
}

Plugin::instance()->widgets_manager->register( new Counter() );
