<?php
/**
 * Contact form 7 timer map
 */

namespace XTS\Elementor;

use Elementor\Widget_Base;
use Elementor\Controls_Manager;
use Elementor\Plugin;

if ( ! defined( 'ABSPATH' ) ) {
	exit; // Direct access not allowed.
}

/**
 * Elementor widget that inserts an embeddable content into the page, from any given URL.
 *
 * @since 1.0.0
 */
class Contact_Form_7 extends Widget_Base {
	/**
	 * Get widget name.
	 *
	 * @since 1.0.0
	 * @access public
	 *
	 * @return string Widget name.
	 */
	public function get_name() {
		return 'wd_contact_form_7';
	}

	/**
	 * Get widget title.
	 *
	 * @since 1.0.0
	 * @access public
	 *
	 * @return string Widget title.
	 */
	public function get_title() {
		return esc_html__( 'Contact form 7', 'studiare-plus' );
	}

	/**
	 * Get widget icon.
	 *
	 * @since 1.0.0
	 * @access public
	 *
	 * @return string Widget icon.
	 */
	public function get_icon() {
		return 'wd-icon-contact-form-7';
	}

	/**
	 * Get widget categories.
	 *
	 * @since 1.0.0
	 * @access public
	 *
	 * @return array Widget categories.
	 */
	public function get_categories() {
		return [ 'wd-elements' ];
	}

	/**
	 * Get contact forms.
	 *
	 * @since 1.0.0
	 * @access public
	 *
	 * @return array Contact forms.
	 */
	public function get_contact_forms() {
		$forms = get_posts(
			[
				'post_type'   => 'wpcf7_contact_form',
				'numberposts' => -1,
			]
		);

		$contact_forms = [];

		if ( $forms ) {
			foreach ( $forms as $form ) {
				$contact_forms[ $form->ID ] = $form->post_title;
			}
		}

		return $contact_forms;
	}

	/**
	 * Register the widget controls.
	 *
	 * @since 1.0.0
	 * @access protected
	 */
	protected function register_controls() {
		/**
		 * Content tab
		 */
		$this->start_controls_section(
			'general_content_section',
			[
				'label' => esc_html__( 'General', 'studiare-plus' ),
			]
		);

		$this->add_control(
			'css_classes',
			array(
				'type'         => 'wd_css_class',
				'default'      => 'wd-cf7',
				'prefix_class' => '',
			)
		);

		$this->add_control(
			'form_id',
			[
				'label'       => esc_html__( 'Select contact form', 'studiare-plus' ),
				'type'        => Controls_Manager::SELECT2,
				'label_block' => true,
				'options'     => $this->get_contact_forms(),
				'default'     => '0',
			]
		);

		$this->end_controls_section();

		/**
		 * Style tab
		 */

		$this->start_controls_section(
			'color_style_section',
			[
				'label' => esc_html__( 'Form', 'studiare-plus' ),
				'tab'   => Controls_Manager::TAB_STYLE,
			]
		);

		$this->add_control(
			'style',
			[
				'label'   => esc_html__( 'Color presets', 'studiare-plus' ),
				'type'    => Controls_Manager::SELECT,
				'options' => [
					''                 => esc_html__( 'Default', 'studiare-plus' ),
					'wd-style-with-bg' => esc_html__( 'With background', 'studiare-plus' ),
				],
				'default' => '',
			]
		);

		$this->add_control(
			'form_color',
			array(
				'label'     => esc_html__( 'Text color', 'studiare-plus' ),
				'type'      => Controls_Manager::COLOR,
				'selectors' => array(
					'{{WRAPPER}} .wpcf7-form' => '--wd-form-color: {{VALUE}};',
				),
				'condition' => array(
					'style!' => 'wd-style-with-bg',
				),
			)
		);

		$this->add_control(
			'form_placeholder_color',
			array(
				'label'     => esc_html__( 'Placeholder color', 'studiare-plus' ),
				'type'      => Controls_Manager::COLOR,
				'selectors' => array(
					'{{WRAPPER}} .wpcf7-form' => '--wd-form-placeholder-color: {{VALUE}};',
				),
				'condition' => array(
					'style!' => 'wd-style-with-bg',
				),
			)
		);

		$this->add_control(
			'form_brd_color',
			array(
				'label'     => esc_html__( 'Border color', 'studiare-plus' ),
				'type'      => Controls_Manager::COLOR,
				'selectors' => array(
					'{{WRAPPER}} .wpcf7-form' => '--wd-form-brd-color: {{VALUE}};',
				),
				'condition' => array(
					'style!' => 'wd-style-with-bg',
				),
			)
		);

		$this->add_control(
			'form_brd_color_focus',
			array(
				'label'     => esc_html__( 'Border color focus', 'studiare-plus' ),
				'type'      => Controls_Manager::COLOR,
				'selectors' => array(
					'{{WRAPPER}} .wpcf7-form' => '--wd-form-brd-color-focus: {{VALUE}};',
				),
				'condition' => array(
					'style!' => 'wd-style-with-bg',
				),
			)
		);

		$this->add_control(
			'form_bg',
			array(
				'label'     => esc_html__( 'Background color', 'studiare-plus' ),
				'type'      => Controls_Manager::COLOR,
				'selectors' => array(
					'{{WRAPPER}} .wpcf7-form' => '--wd-form-bg: {{VALUE}};',
				),
				'condition' => array(
					'style!' => 'wd-style-with-bg',
				),
			)
		);

		$this->end_controls_section();

		$this->start_controls_section(
			'button_color_style_section',
			[
				'label' => esc_html__( 'Button', 'studiare-plus' ),
				'tab'   => Controls_Manager::TAB_STYLE,
			]
		);

		$this->start_controls_tabs( 'color_tabs' );

		$this->start_controls_tab(
			'title_idle_color_tab',
			array(
				'label' => esc_html__( 'Idle', 'studiare-plus' ),
			)
		);

		$this->add_control(
			'button_text_color',
			array(
				'label'     => esc_html__( 'Color', 'studiare-plus' ),
				'type'      => Controls_Manager::COLOR,
				'selectors' => array(
					'{{WRAPPER}} .wpcf7-submit' => '--btn-accented-color: {{VALUE}};',
				),
			)
		);

		$this->add_control(
			'button_bg_color',
			array(
				'label'     => esc_html__( 'Background color', 'studiare-plus' ),
				'type'      => Controls_Manager::COLOR,
				'selectors' => array(
					'{{WRAPPER}} .wpcf7-submit' => '--btn-accented-bgcolor: {{VALUE}};',
				),
			)
		);

		$this->end_controls_tab();

		$this->start_controls_tab(
			'title_hover_color_tab',
			array(
				'label' => esc_html__( 'Hover', 'studiare-plus' ),
			)
		);

		$this->add_control(
			'button_text_color_hover',
			array(
				'label'     => esc_html__( 'Color', 'studiare-plus' ),
				'type'      => Controls_Manager::COLOR,
				'selectors' => array(
					'{{WRAPPER}} .wpcf7-submit' => '--btn-accented-color-hover: {{VALUE}};',
				),
			)
		);

		$this->add_control(
			'button_bg_color_hover',
			array(
				'label'     => esc_html__( 'Background color', 'studiare-plus' ),
				'type'      => Controls_Manager::COLOR,
				'selectors' => array(
					'{{WRAPPER}} .wpcf7-submit' => '--btn-accented-bgcolor-hover: {{VALUE}};',
				),
			)
		);

		$this->end_controls_tab();

		$this->end_controls_tabs();

		$this->end_controls_section();
	}

	/**
	 * Render the widget output on the frontend.
	 *
	 * Written in PHP and used to generate the final HTML.
	 *
	 * @since 1.0.0
	 *
	 * @access protected
	 */
	protected function render() {
		$default_settings = [
			'form_id' => '0',
			'style'   => '',
		];

		$settings = wp_parse_args( $this->get_settings_for_display(), $default_settings );

		woodmart_enqueue_inline_style( 'wpcf7', true );

		if ( ! $settings['form_id'] || ! defined( 'WPCF7_PLUGIN' ) ) {
			echo '<div class="wd-notice wd-info">' . esc_html__( 'You need to create a form using Contact form 7 plugin to be able to display it using this element.', 'studiare-plus' ) . '</div>';
			return;
		}

		if ( function_exists( 'wpcf7_enqueue_scripts' ) && ! wp_script_is( 'contact-form-7', 'registered' ) ) {
			$assets = include wpcf7_plugin_path( 'includes/js/index.asset.php' );

			$assets = wp_parse_args(
				$assets,
				array(
					'dependencies' => array(),
					'version'      => WPCF7_VERSION,
				)
			);

			wp_register_script(
				'contact-form-7',
				wpcf7_plugin_url( 'includes/js/index.js' ),
				array_merge(
					$assets['dependencies'],
					array( 'swv' )
				),
				$assets['version'],
				array( 'in_footer' => true )
			);

			wpcf7_enqueue_scripts();
		}

		echo do_shortcode( '[contact-form-7 html_class="' . esc_attr( $settings['style'] ) . '" id="' . esc_attr( $settings['form_id'] ) . '"]' );
	}
}

Plugin::instance()->widgets_manager->register( new Contact_Form_7() );
