<?php
/**
 * Promo banner map.
 *
 * @package studiare plus
 */

namespace XTS\Elementor;

use Elementor\Group_Control_Background;
use Elementor\Group_Control_Image_Size;
use Elementor\Group_Control_Typography;
use Elementor\Modules\DynamicTags\Module as TagsModule;
use Elementor\Utils;
use Elementor\Widget_Base;
use Elementor\Controls_Manager;
use Elementor\Group_Control_Box_Shadow;
use Elementor\Plugin;

if ( ! defined( 'ABSPATH' ) ) {
	exit; // Direct access not allowed.
}

/**
 * Elementor widget that inserts an embeddable content into the page, from any given URL.
 *
 * @since 1.0.0
 */
class Banner extends Widget_Base {
	/**
	 * Get widget name.
	 *
	 * @since 1.0.0
	 * @access public
	 *
	 * @return string Widget name.
	 */
	public function get_name() {
		return 'wd_banner';
	}

	/**
	 * Get widget title.
	 *
	 * @since 1.0.0
	 * @access public
	 *
	 * @return string Widget title.
	 */
	public function get_title() {
		return esc_html__( 'Promo Banner', 'studiare-plus' );
	}

	/**
	 * Get widget icon.
	 *
	 * @since 1.0.0
	 * @access public
	 *
	 * @return string Widget icon.
	 */
	public function get_icon() {
		return 'wd-icon-banner';
	}

	/**
	 * Get widget categories.
	 *
	 * @since 1.0.0
	 * @access public
	 *
	 * @return array Widget categories.
	 */
	public function get_categories() {
		return array( 'wd-elements' );
	}

	/**
	 * Register the widget controls.
	 *
	 * @since 1.0.0
	 * @access protected
	 */
	protected function register_controls() {
		/**
		 * Content tab.
		 */

		/**
		 * General settings.
		 */
		$this->start_controls_section(
			'general_content_section',
			array(
				'label' => esc_html__( 'General', 'studiare-plus' ),
			)
		);

		$this->add_control(
			'link',
			array(
				'label'       => esc_html__( 'Link', 'studiare-plus' ),
				'description' => esc_html__( 'Enter URL if you want this banner to have a link.', 'studiare-plus' ),
				'type'        => Controls_Manager::URL,
				'default'     => array(
					'url'         => '#',
					'is_external' => false,
					'nofollow'    => false,
				),
			)
		);

		$this->end_controls_section();

		/**
		 * Image settings.
		 */
		$this->start_controls_section(
			'image_content_section',
			array(
				'label' => esc_html__( 'Background', 'studiare-plus' ),
			)
		);

		$this->add_control(
			'source_type',
			array(
				'label'   => esc_html__( 'Source', 'studiare-plus' ),
				'type'    => Controls_Manager::SELECT,
				'options' => array(
					'image' => esc_html__( 'Image', 'studiare-plus' ),
					'video' => esc_html__( 'Video', 'studiare-plus' ),
				),
				'default' => 'image',
			)
		);

		$this->add_control(
			'video',
			array(
				'label'      => esc_html__( 'Choose video', 'studiare-plus' ),
				'type'       => Controls_Manager::MEDIA,
				'media_type' => 'video',
				'condition'  => array(
					'source_type' => 'video',
				),
			)
		);

		$this->add_control(
			'video_poster',
			array(
				'label'     => esc_html__( 'Fallback image', 'studiare-plus' ),
				'type'      => Controls_Manager::MEDIA,
				'condition' => array(
					'source_type' => 'video',
				),
			)
		);

		$this->add_group_control(
			Group_Control_Image_Size::get_type(),
			array(
				'name'           => 'video_poster',
				'fields_options' => array(
					'size'             => array(
						'label' => esc_html__( 'Fallback image size', 'studiare-plus' ),
					),
					'custom_dimension' => array(
						'label' => esc_html__( 'Fallback image Dimension', 'studiare-plus' ),
					),
				),
				'default'        => 'full',
				'separator'      => 'none',
				'condition'      => array(
					'source_type' => 'video',
				),
			)
		);

		$this->add_control(
			'image',
			array(
				'label'     => esc_html__( 'Choose image', 'studiare-plus' ),
				'type'      => Controls_Manager::MEDIA,
				'default'   => array(
					'url' => Utils::get_placeholder_image_src(),
				),
				'condition' => array(
					'source_type' => 'image',
				),
			)
		);

		$this->add_group_control(
			Group_Control_Image_Size::get_type(),
			array(
				'name'      => 'image',
				'default'   => 'thumbnail',
				'separator' => 'none',
				'condition' => array(
					'source_type' => 'image',
				),
			)
		);

		$this->add_control(
			'custom_height',
			array(
				'label'        => esc_html__( 'Fixed height', 'studiare-plus' ),
				'type'         => Controls_Manager::SWITCHER,
				'default'      => 'no',
				'label_on'     => esc_html__( 'Yes', 'studiare-plus' ),
				'label_off'    => esc_html__( 'No', 'studiare-plus' ),
				'return_value' => 'Yes',
			)
		);

		$this->add_responsive_control(
			'image_height',
			array(
				'label'     => esc_html__( 'Banner height', 'studiare-plus' ),
				'type'      => Controls_Manager::SLIDER,
				'default'   => array(
					'size' => 340,
				),
				'range'     => array(
					'px' => array(
						'min'  => 100,
						'max'  => 2000,
						'step' => 1,
					),
				),
				'selectors' => array(
					'{{WRAPPER}}' => '--wd-img-height: {{SIZE}}{{UNIT}};',
				),
				'condition' => array(
					'custom_height' => array( 'Yes' ),
				),
			)
		);

		$this->add_control(
			'image_bg_position',
			array(
				'label'     => esc_html__( 'Image Position', 'studiare-plus' ),
				'type'      => Controls_Manager::SELECT,
				'options'   => array(
					'center center' => esc_html__( 'Center', 'studiare-plus' ),
					'center top'    => esc_html__( 'Top', 'studiare-plus' ),
					'center bottom' => esc_html__( 'Bottom', 'studiare-plus' ),
					'left center'   => esc_html__( 'Left', 'studiare-plus' ),
					'right center'  => esc_html__( 'Right', 'studiare-plus' ),
				),
				'default'   => 'center center',
				'selectors' => array(
					'{{WRAPPER}} .banner-image' => 'object-position: {{VALUE}};',
				),
				'condition' => array(
					'custom_height' => array( 'Yes' ),
					'source_type'   => 'image',
				),
			)
		);

		$this->end_controls_section();

		/**
		 * Content settings.
		 */
		$this->start_controls_section(
			'content_section',
			array(
				'label' => esc_html__( 'Content', 'studiare-plus' ),
			)
		);

		$this->add_control(
			'subtitle',
			array(
				'label'   => esc_html__( 'Subtitle', 'studiare-plus' ),
				'type'    => Controls_Manager::TEXTAREA,
				'default' => 'Banner subtitle text',
			)
		);

		$this->add_control(
			'title',
			array(
				'label'   => esc_html__( 'Title', 'studiare-plus' ),
				'type'    => Controls_Manager::TEXTAREA,
				'default' => 'Banner title, click to edit.',
			)
		);

		$this->add_control(
			'content',
			array(
				'label'   => esc_html__( 'Content', 'studiare-plus' ),
				'type'    => Controls_Manager::WYSIWYG,
				'default' => 'Banner content text',
			)
		);

		$this->add_control(
			'btn_text',
			array(
				'label'   => esc_html__( 'Button text', 'studiare-plus' ),
				'type'    => Controls_Manager::TEXT,
				'default' => 'Read more',
			)
		);

		$this->end_controls_section();

		/**
		 * Countdown settings.
		 */
		$this->start_controls_section(
			'countdown_section',
			array(
				'label' => esc_html__( 'Countdown', 'studiare-plus' ),
			)
		);

		$this->add_control(
			'show_countdown',
			array(
				'label'        => esc_html__( 'Show countdown', 'studiare-plus' ),
				'type'         => Controls_Manager::SWITCHER,
				'default'      => 'no',
				'label_on'     => esc_html__( 'Yes', 'studiare-plus' ),
				'label_off'    => esc_html__( 'No', 'studiare-plus' ),
				'return_value' => 'yes',
			)
		);

		$this->add_control(
			'date',
			array(
				'label'     => esc_html__( 'Date', 'studiare-plus' ),
				'type'      => Controls_Manager::DATE_TIME,
				'default'   => date( 'Y-m-d', strtotime( ' +2 months' ) ),
				'condition' => array(
					'show_countdown' => array( 'yes' ),
				),
			)
		);

		$this->add_control(
			'hide_countdown_on_finish',
			array(
				'label'        => esc_html__( 'Hide countdown on finish', 'studiare-plus' ),
				'type'         => Controls_Manager::SWITCHER,
				'default'      => 'no',
				'label_on'     => esc_html__( 'Yes', 'studiare-plus' ),
				'label_off'    => esc_html__( 'No', 'studiare-plus' ),
				'return_value' => 'yes',
				'condition'    => array(
					'show_countdown' => array( 'yes' ),
				),
			)
		);

		$this->end_controls_section();

		/**
		 * Style tab.
		 */

		/**
		 * General settings.
		 */
		$this->start_controls_section(
			'general_style_section',
			array(
				'label' => esc_html__( 'General', 'studiare-plus' ),
				'tab'   => Controls_Manager::TAB_STYLE,
			)
		);

		$this->add_control(
			'style',
			array(
				'label'       => esc_html__( 'Style', 'studiare-plus' ),
				'description' => esc_html__( 'You can use some of our predefined styles for your banner content.', 'studiare-plus' ),
				'type'        => Controls_Manager::SELECT,
				'options'     => array(
					'default'            => esc_html__( 'Default', 'studiare-plus' ),
					'mask'               => esc_html__( 'Color mask', 'studiare-plus' ),
					'shadow'             => esc_html__( 'Mask with shadow', 'studiare-plus' ),
					'border'             => esc_html__( 'Bordered', 'studiare-plus' ),
					'background'         => esc_html__( 'Bordered background', 'studiare-plus' ),
					'content-background' => esc_html__( 'Content background', 'studiare-plus' ),
				),
				'default'     => 'default',
			)
		);

		$this->add_control(
			'hover',
			array(
				'label'       => esc_html__( 'Hover effect', 'studiare-plus' ),
				'description' => esc_html__( 'Set beautiful hover effects for your banner.', 'studiare-plus' ),
				'type'        => Controls_Manager::SELECT,
				'options'     => array(
					'zoom'         => esc_html__( 'Zoom image', 'studiare-plus' ),
					'parallax'     => esc_html__( 'Parallax', 'studiare-plus' ),
					'background'   => esc_html__( 'Background', 'studiare-plus' ),
					'border'       => esc_html__( 'Bordered', 'studiare-plus' ),
					'zoom-reverse' => esc_html__( 'Zoom reverse', 'studiare-plus' ),
					'none'         => esc_html__( 'Disable', 'studiare-plus' ),
				),
				'default'     => 'none',
			)
		);

		$this->add_control(
			'rounding_size',
			array(
				'label'     => esc_html__( 'Rounding', 'studiare-plus' ),
				'type'      => Controls_Manager::SELECT,
				'options'   => array(
					''       => esc_html__( 'Inherit', 'studiare-plus' ),
					'0'      => esc_html__( '0', 'studiare-plus' ),
					'5'      => esc_html__( '5', 'studiare-plus' ),
					'8'      => esc_html__( '8', 'studiare-plus' ),
					'12'     => esc_html__( '12', 'studiare-plus' ),
					'custom' => esc_html__( 'Custom', 'studiare-plus' ),
				),
				'default'   => '',
				'selectors' => array(
					'{{WRAPPER}}' => '--wd-brd-radius: {{VALUE}}px;',
				),
			)
		);

		$this->add_control(
			'custom_rounding_size',
			array(
				'label'      => esc_html__( 'Custom rounding', 'studiare-plus' ),
				'type'       => Controls_Manager::SLIDER,
				'size_units' => array( '%', 'px' ),
				'range'      => array(
					'px' => array(
						'min'  => 1,
						'max'  => 300,
						'step' => 1,
					),
					'%'  => array(
						'min'  => 1,
						'max'  => 100,
						'step' => 1,
					),
				),
				'selectors'  => array(
					'{{WRAPPER}}' => '--wd-brd-radius: {{SIZE}}{{UNIT}};',
				),
				'condition'  => array(
					'rounding_size' => array( 'custom' ),
				),
			)
		);

		$this->add_control(
			'woodmart_color_scheme',
			array(
				'label'   => esc_html__( 'Color Scheme', 'studiare-plus' ),
				'type'    => Controls_Manager::SELECT,
				'options' => array(
					''      => esc_html__( 'Inherit', 'studiare-plus' ),
					'light' => esc_html__( 'Light', 'studiare-plus' ),
					'dark'  => esc_html__( 'Dark', 'studiare-plus' ),
				),
				'default' => '',
			)
		);

		$this->add_control(
			'title_size',
			array(
				'label'   => esc_html__( 'Predefined size', 'studiare-plus' ),
				'type'    => Controls_Manager::SELECT,
				'options' => array(
					'default'     => esc_html__( 'Default (22px)', 'studiare-plus' ),
					'small'       => esc_html__( 'Small (16px)', 'studiare-plus' ),
					'large'       => esc_html__( 'Large (26px)', 'studiare-plus' ),
					'extra-large' => esc_html__( 'Extra Large (36px)', 'studiare-plus' ),
				),
				'default' => 'default',
			)
		);

		$this->add_control(
			'custom_content_bg_color',
			array(
				'label'     => esc_html__( 'Content background color', 'studiare-plus' ),
				'type'      => Controls_Manager::COLOR,
				'selectors' => array(
					'{{WRAPPER}} .wrapper-content-banner' => 'background-color: {{VALUE}}',
				),
				'condition' => array(
					'style' => 'content-background',
				),
			)
		);

		$this->end_controls_section();

		/**
		 * Title settings.
		 */
		$this->start_controls_section(
			'title_style_section',
			array(
				'label' => esc_html__( 'Title', 'studiare-plus' ),
				'tab'   => Controls_Manager::TAB_STYLE,
			)
		);

		$this->add_control(
			'title_tag',
			array(
				'label'   => esc_html__( 'Tag', 'studiare-plus' ),
				'type'    => Controls_Manager::SELECT,
				'options' => array(
					'h1'   => esc_html__( 'h1', 'studiare-plus' ),
					'h2'   => esc_html__( 'h2', 'studiare-plus' ),
					'h3'   => esc_html__( 'h3', 'studiare-plus' ),
					'h4'   => esc_html__( 'h4', 'studiare-plus' ),
					'h5'   => esc_html__( 'h5', 'studiare-plus' ),
					'h6'   => esc_html__( 'h6', 'studiare-plus' ),
					'p'    => esc_html__( 'p', 'studiare-plus' ),
					'div'  => esc_html__( 'div', 'studiare-plus' ),
					'span' => esc_html__( 'span', 'studiare-plus' ),
				),
				'default' => 'h4',
			)
		);

		$this->add_control(
			'custom_title_color',
			array(
				'label'     => esc_html__( 'Color', 'studiare-plus' ),
				'type'      => Controls_Manager::COLOR,
				'selectors' => array(
					'{{WRAPPER}} .banner-title' => 'color: {{VALUE}}',
				),
			)
		);

		$this->add_group_control(
			Group_Control_Typography::get_type(),
			array(
				'name'     => 'title_typography',
				'label'    => esc_html__( 'Custom typography', 'studiare-plus' ),
				'selector' => '{{WRAPPER}} .banner-title',
			)
		);

		$this->add_control(
			'title_decoration_style',
			array(
				'label'       => esc_html__( 'Highlight text style', 'studiare-plus' ),
				'description' => esc_html__( 'The text must be wrapped with the <u></u> tag to highlight it.', 'studiare-plus' ),
				'type'        => Controls_Manager::SELECT,
				'options'     => array(
					'default'     => esc_html__( 'Default', 'studiare-plus' ),
					'colored'     => esc_html__( 'Primary color', 'studiare-plus' ),
					'colored-alt' => esc_html__( 'Primary color + secondary font', 'studiare-plus' ),
					'bordered'    => esc_html__( 'Bordered', 'studiare-plus' ),
					'gradient'    => esc_html__( 'Gradient', 'studiare-plus' ),
				),
				'default'     => 'default',
			)
		);

		$this->add_group_control(
			Group_Control_Background::get_type(),
			array(
				'name'      => 'title_decoration_gradient',
				'types'     => array( 'gradient' ),
				'selector'  => '{{WRAPPER}} .wd-underline-gradient u',
				'condition' => array(
					'title_decoration_style' => 'gradient',
				),
			)
		);

		$this->end_controls_section();

		/**
		 * Subtitle settings.
		 */
		$this->start_controls_section(
			'subtitle_style_section',
			array(
				'label' => esc_html__( 'Subtitle', 'studiare-plus' ),
				'tab'   => Controls_Manager::TAB_STYLE,
			)
		);

		$this->add_control(
			'subtitle_style',
			array(
				'label'   => esc_html__( 'Subtitle style', 'studiare-plus' ),
				'type'    => Controls_Manager::SELECT,
				'options' => array(
					'default'    => esc_html__( 'Default', 'studiare-plus' ),
					'background' => esc_html__( 'Background', 'studiare-plus' ),
				),
				'default' => 'default',
			)
		);

		$this->add_control(
			'subtitle_color',
			array(
				'label'   => esc_html__( 'Predefined color', 'studiare-plus' ),
				'type'    => Controls_Manager::SELECT,
				'options' => array(
					'default' => esc_html__( 'Default', 'studiare-plus' ),
					'primary' => esc_html__( 'Primary', 'studiare-plus' ),
					'alt'     => esc_html__( 'Alternative', 'studiare-plus' ),
				),
				'default' => 'default',
			)
		);

		$this->add_control(
			'custom_subtitle_color',
			array(
				'label'     => esc_html__( 'Color', 'studiare-plus' ),
				'type'      => Controls_Manager::COLOR,
				'selectors' => array(
					'{{WRAPPER}} .banner-subtitle' => 'color: {{VALUE}}',
				),
			)
		);

		$this->add_control(
			'custom_subtitle_bg_color',
			array(
				'label'     => esc_html__( 'Background color', 'studiare-plus' ),
				'type'      => Controls_Manager::COLOR,
				'selectors' => array(
					'{{WRAPPER}} .banner-subtitle' => 'background-color: {{VALUE}}',
				),
				'condition' => array(
					'subtitle_style' => array( 'background' ),
				),
			)
		);

		$this->add_group_control(
			Group_Control_Typography::get_type(),
			array(
				'name'     => 'subtitle_typography',
				'label'    => esc_html__( 'Custom typography', 'studiare-plus' ),
				'selector' => '{{WRAPPER}} .banner-subtitle',
			)
		);

		$this->end_controls_section();

		/**
		 * Content settings.
		 */
		$this->start_controls_section(
			'content_style_section',
			array(
				'label' => esc_html__( 'Content', 'studiare-plus' ),
				'tab'   => Controls_Manager::TAB_STYLE,
			)
		);

		$this->add_control(
			'content_text_size',
			array(
				'label'   => esc_html__( 'Predefined size', 'studiare-plus' ),
				'type'    => Controls_Manager::SELECT,
				'options' => array(
					'default' => esc_html__( 'Default (14px)', 'studiare-plus' ),
					'medium'  => esc_html__( 'Medium (16px)', 'studiare-plus' ),
					'large'   => esc_html__( 'Large (18px)', 'studiare-plus' ),
				),
				'default' => 'default',
			)
		);

		$this->add_control(
			'custom_text_color',
			array(
				'label'     => esc_html__( 'Color', 'studiare-plus' ),
				'type'      => Controls_Manager::COLOR,
				'selectors' => array(
					'{{WRAPPER}} .banner-inner' => 'color: {{VALUE}}',
				),
			)
		);

		$this->add_group_control(
			Group_Control_Typography::get_type(),
			array(
				'name'     => 'text_typography',
				'label'    => esc_html__( 'Custom typography', 'studiare-plus' ),
				'selector' => '{{WRAPPER}} .banner-inner',
			)
		);

		$this->end_controls_section();

		/**
		 * Countdown settings.
		 */
		$this->start_controls_section(
			'countdown_style_section',
			array(
				'label' => esc_html__( 'Countdown', 'studiare-plus' ),
				'tab'   => Controls_Manager::TAB_STYLE,
			)
		);

		$this->add_control(
			'countdown_style',
			array(
				'label'   => esc_html__( 'Background', 'studiare-plus' ),
				'type'    => Controls_Manager::SELECT,
				'options' => [
					'simple' => esc_html__( 'Default', 'studiare-plus' ),
					'active' => esc_html__( 'Primary color', 'studiare-plus' ),
					'custom' => esc_html__( 'Custom', 'studiare-plus' ),
				],
				'default' => 'simple',
			)
		);

		$this->add_control(
			'countdown_bg_color',
			array(
				'label'     => esc_html__( 'Background color', 'studiare-plus' ),
				'type'      => Controls_Manager::COLOR,
				'selectors' => array(
					'{{WRAPPER}} .wd-timer' => '--wd-timer-bg: {{VALUE}}',
				),
				'condition' => array(
					'countdown_style' => 'custom',
				),
			)
		);

		$this->add_control(
			'countdown_color_scheme',
			array(
				'label'   => esc_html__( 'Color Scheme', 'studiare-plus' ),
				'type'    => Controls_Manager::SELECT,
				'options' => array(
					''      => esc_html__( 'Inherit', 'studiare-plus' ),
					'light' => esc_html__( 'Light', 'studiare-plus' ),
					'dark'  => esc_html__( 'Dark', 'studiare-plus' ),
				),
				'default' => '',
			)
		);

		$this->add_group_control(
			Group_Control_Box_Shadow::get_type(),
			array(
				'name'     => 'countdown_shadow',
				'selector' => '{{WRAPPER}} .wd-item',
			)
		);

		$this->add_control(
			'countdown_size',
			array(
				'label'   => esc_html__( 'Predefined size', 'studiare-plus' ),
				'type'    => Controls_Manager::SELECT,
				'options' => array(
					'small'  => esc_html__( 'Small (20px)', 'studiare-plus' ),
					'medium' => esc_html__( 'Medium (24px)', 'studiare-plus' ),
					'large'  => esc_html__( 'Large (28px)', 'studiare-plus' ),
					'xlarge' => esc_html__( 'Extra Large (42px)', 'studiare-plus' ),
				),
				'default' => 'medium',
			)
		);

		$this->end_controls_section();

		/**
		 * Button settings.
		 */
		$this->start_controls_section(
			'button_style_section',
			array(
				'label' => esc_html__( 'Button', 'studiare-plus' ),
				'tab'   => Controls_Manager::TAB_STYLE,
			)
		);

		$this->add_control(
			'btn_position',
			array(
				'label'     => esc_html__( 'Button position', 'studiare-plus' ),
				'type'      => Controls_Manager::SELECT,
				'options'   => array(
					'hover'  => esc_html__( 'Show on hover', 'studiare-plus' ),
					'static' => esc_html__( 'Static', 'studiare-plus' ),
				),
				'default'   => 'hover',
				'condition' => array(
					'style!' => 'content-background',
				),
			)
		);

		$this->add_control(
			'btn_style',
			array(
				'label'   => esc_html__( 'Style', 'studiare-plus' ),
				'type'    => Controls_Manager::SELECT,
				'options' => array(
					'default'  => esc_html__( 'Flat', 'studiare-plus' ),
					'bordered' => esc_html__( 'Bordered', 'studiare-plus' ),
					'link'     => esc_html__( 'Link button', 'studiare-plus' ),
					'3d'       => esc_html__( '3D', 'studiare-plus' ),
				),
				'default' => 'default',
			)
		);

		$this->add_control(
			'btn_color',
			array(
				'label'   => esc_html__( 'Predefined color', 'studiare-plus' ),
				'type'    => Controls_Manager::SELECT,
				'options' => array(
					'default' => esc_html__( 'Default', 'studiare-plus' ),
					'primary' => esc_html__( 'Primary', 'studiare-plus' ),
					'alt'     => esc_html__( 'Alternative', 'studiare-plus' ),
					'black'   => esc_html__( 'Black', 'studiare-plus' ),
					'white'   => esc_html__( 'White', 'studiare-plus' ),
					'custom'  => esc_html__( 'Custom', 'studiare-plus' ),
				),
				'default' => 'default',
			)
		);

		$this->start_controls_tabs(
			'btn_tabs_style',
			array(
				'condition' => array(
					'btn_color' => array( 'custom' ),
				),
			)
		);

		$this->start_controls_tab(
			'btn_tab_normal',
			array(
				'label' => esc_html__( 'Normal', 'studiare-plus' ),
			)
		);

		$this->add_control(
			'btn_bg_color',
			array(
				'label'     => esc_html__( 'Background color', 'studiare-plus' ),
				'type'      => Controls_Manager::COLOR,
				'selectors' => array(
					'{{WRAPPER}} .wd-button-wrapper a' => 'background-color: {{VALUE}}; border-color: {{VALUE}};',
				),
			)
		);

		$this->add_control(
			'btn_color_scheme',
			array(
				'label'   => esc_html__( 'Text color scheme', 'studiare-plus' ),
				'type'    => Controls_Manager::SELECT,
				'options' => array(
					'inherit' => esc_html__( 'Inherit', 'studiare-plus' ),
					'dark'    => esc_html__( 'Dark', 'studiare-plus' ),
					'light'   => esc_html__( 'Light', 'studiare-plus' ),
					'custom'  => esc_html__( 'Custom', 'studiare-plus' ),
				),
				'default' => 'inherit',
			)
		);

		$this->add_control(
			'btn_custom_color_scheme',
			array(
				'label'     => esc_html__( 'Custom text color', 'studiare-plus' ),
				'type'      => Controls_Manager::COLOR,
				'selectors' => array(
					'{{WRAPPER}} .wd-button-wrapper a' => 'color: {{VALUE}};',
				),
				'condition' => array(
					'btn_color_scheme' => 'custom',
				),
			)
		);

		$this->end_controls_tab();

		$this->start_controls_tab(
			'btn_tab_hover',
			array(
				'label' => esc_html__( 'Hover', 'studiare-plus' ),
			)
		);

		$this->add_control(
			'btn_bg_color_hover',
			array(
				'label'     => esc_html__( 'Background color hover', 'studiare-plus' ),
				'type'      => Controls_Manager::COLOR,
				'selectors' => array(
					'{{WRAPPER}} .wd-button-wrapper:hover a' => 'background-color: {{VALUE}}; border-color: {{VALUE}};',
				),
			)
		);

		$this->add_control(
			'btn_color_scheme_hover',
			array(
				'label'   => esc_html__( 'Text color scheme on hover', 'studiare-plus' ),
				'type'    => Controls_Manager::SELECT,
				'options' => array(
					'inherit' => esc_html__( 'Inherit', 'studiare-plus' ),
					'dark'    => esc_html__( 'Dark', 'studiare-plus' ),
					'light'   => esc_html__( 'Light', 'studiare-plus' ),
					'custom'  => esc_html__( 'Custom', 'studiare-plus' ),
				),
				'default' => 'inherit',
			)
		);

		$this->add_control(
			'btn_custom_color_scheme_hover',
			array(
				'label'     => esc_html__( 'Custom text color on hover', 'studiare-plus' ),
				'type'      => Controls_Manager::COLOR,
				'selectors' => array(
					'{{WRAPPER}} .wd-button-wrapper:hover a' => 'color: {{VALUE}};',
				),
				'condition' => array(
					'btn_color_scheme_hover' => 'custom',
				),
			)
		);

		$this->end_controls_tab();

		$this->end_controls_tabs();

		$this->add_control(
			'btn_size',
			array(
				'label'   => esc_html__( 'Predefined size', 'studiare-plus' ),
				'type'    => Controls_Manager::SELECT,
				'options' => array(
					'default'     => esc_html__( 'Default', 'studiare-plus' ),
					'extra-small' => esc_html__( 'Extra Small', 'studiare-plus' ),
					'small'       => esc_html__( 'Small', 'studiare-plus' ),
					'large'       => esc_html__( 'Large', 'studiare-plus' ),
					'extra-large' => esc_html__( 'Extra Large', 'studiare-plus' ),
				),
				'default' => 'default',
			)
		);

		$this->add_control(
			'btn_shape',
			array(
				'label'     => esc_html__( 'Shape', 'studiare-plus' ),
				'type'      => Controls_Manager::SELECT,
				'options'   => array(
					'rectangle'  => esc_html__( 'Rectangle', 'studiare-plus' ),
					'round'      => esc_html__( 'Round', 'studiare-plus' ),
					'semi-round' => esc_html__( 'Rounded', 'studiare-plus' ),
				),
				'condition' => array(
					'btn_style!' => array( 'link' ),
				),
				'default'   => 'rectangle',
			)
		);

		$this->add_control(
			'button_icon_heading',
			array(
				'label'     => esc_html__( 'Icon', 'studiare-plus' ),
				'type'      => Controls_Manager::HEADING,
				'separator' => 'before',
			)
		);

		woodmart_get_button_style_icon_map( $this, 'btn_' );

		$this->add_control(
			'button_layout_heading',
			array(
				'label'     => esc_html__( 'Layout', 'studiare-plus' ),
				'type'      => Controls_Manager::HEADING,
				'separator' => 'before',
			)
		);

		$this->add_control(
			'full_width',
			array(
				'label'        => esc_html__( 'Full width', 'studiare-plus' ),
				'type'         => Controls_Manager::SWITCHER,
				'default'      => 'no',
				'label_on'     => esc_html__( 'Yes', 'studiare-plus' ),
				'label_off'    => esc_html__( 'No', 'studiare-plus' ),
				'return_value' => 'yes',
			)
		);

		$this->add_control(
			'button_hr',
			array(
				'type'  => Controls_Manager::DIVIDER,
				'style' => 'thick',
			)
		);

		$this->add_control(
			'hide_btn_tablet',
			array(
				'label'        => esc_html__( 'Hide button on tablet', 'studiare-plus' ),
				'type'         => Controls_Manager::SWITCHER,
				'default'      => 'no',
				'label_on'     => esc_html__( 'Yes', 'studiare-plus' ),
				'label_off'    => esc_html__( 'No', 'studiare-plus' ),
				'return_value' => 'yes',
			)
		);

		$this->add_control(
			'hide_btn_mobile',
			array(
				'label'        => esc_html__( 'Hide button on mobile', 'studiare-plus' ),
				'type'         => Controls_Manager::SWITCHER,
				'default'      => 'no',
				'label_on'     => esc_html__( 'Yes', 'studiare-plus' ),
				'label_off'    => esc_html__( 'No', 'studiare-plus' ),
				'return_value' => 'yes',
			)
		);

		$this->end_controls_section();

		/**
		 * Layout settings.
		 */
		$this->start_controls_section(
			'layout_style_section',
			array(
				'label' => esc_html__( 'Layout', 'studiare-plus' ),
				'tab'   => Controls_Manager::TAB_STYLE,
			)
		);

		$this->add_control(
			'horizontal_alignment',
			array(
				'label'   => esc_html__( 'Content horizontal alignment', 'studiare-plus' ),
				'type'    => 'wd_buttons',
				'options' => array(
					'left'   => array(
						'title' => esc_html__( 'Left', 'studiare-plus' ),
						'image' => WOODMART_ASSETS_IMAGES . '/settings/content-align/horizontal/left.png',
					),
					'center' => array(
						'title' => esc_html__( 'Center', 'studiare-plus' ),
						'image' => WOODMART_ASSETS_IMAGES . '/settings/content-align/horizontal/center.png',
					),
					'right'  => array(
						'title' => esc_html__( 'Right', 'studiare-plus' ),
						'image' => WOODMART_ASSETS_IMAGES . '/settings/content-align/horizontal/right.png',
					),

				),
				'default' => 'left',
			)
		);

		$this->add_control(
			'vertical_alignment',
			array(
				'label'   => esc_html__( 'Content vertical alignment', 'studiare-plus' ),
				'type'    => 'wd_buttons',
				'options' => array(
					'top'    => array(
						'title' => esc_html__( 'Top', 'studiare-plus' ),
						'image' => WOODMART_ASSETS_IMAGES . '/settings/content-align/vertical/top.png',
					),
					'middle' => array(
						'title' => esc_html__( 'Middle', 'studiare-plus' ),
						'image' => WOODMART_ASSETS_IMAGES . '/settings/content-align/vertical/middle.png',
					),
					'bottom' => array(
						'title' => esc_html__( 'Bottom', 'studiare-plus' ),
						'image' => WOODMART_ASSETS_IMAGES . '/settings/content-align/vertical/bottom.png',
					),
				),
				'default' => 'top',
			)
		);

		$this->add_control(
			'text_alignment',
			array(
				'label'   => esc_html__( 'Text alignment', 'studiare-plus' ),
				'type'    => 'wd_buttons',
				'options' => array(
					'left'   => array(
						'title' => esc_html__( 'Left', 'studiare-plus' ),
						'image' => WOODMART_ASSETS_IMAGES . '/settings/align/left.jpg',
					),
					'center' => array(
						'title' => esc_html__( 'Center', 'studiare-plus' ),
						'image' => WOODMART_ASSETS_IMAGES . '/settings/align/center.jpg',
					),
					'right'  => array(
						'title' => esc_html__( 'Right', 'studiare-plus' ),
						'image' => WOODMART_ASSETS_IMAGES . '/settings/align/right.jpg',
					),
				),
				'default' => 'left',
			)
		);

		$this->add_responsive_control(
			'width',
			array(
				'label'          => esc_html__( 'Width', 'studiare-plus' ),
				'type'           => Controls_Manager::SLIDER,
				'default'        => array(
					'unit' => '%',
				),
				'tablet_default' => array(
					'unit' => '%',
				),
				'mobile_default' => array(
					'unit' => '%',
				),
				'size_units'     => array( '%', 'px' ),
				'range'          => array(
					'%'  => array(
						'min' => 1,
						'max' => 100,
					),
					'px' => array(
						'min' => 1,
						'max' => 1000,
					),
				),
				'selectors'      => array(
					'{{WRAPPER}} .promo-banner:not(.banner-content-background) .content-banner, {{WRAPPER}} .promo-banner.banner-content-background .wrapper-content-banner' => 'max-width: {{SIZE}}{{UNIT}};',
				),
			)
		);

		$this->add_responsive_control(
			'content_height',
			array(
				'label'          => esc_html__( 'Height', 'studiare-plus' ),
				'type'           => Controls_Manager::SLIDER,
				'default'        => array(
					'unit' => '%',
				),
				'tablet_default' => array(
					'unit' => '%',
				),
				'mobile_default' => array(
					'unit' => '%',
				),
				'size_units'     => array( '%', 'px' ),
				'range'          => array(
					'%'  => array(
						'min' => 1,
						'max' => 100,
					),
					'px' => array(
						'min' => 1,
						'max' => 1000,
					),
				),
				'selectors'      => array(
					'{{WRAPPER}} .promo-banner:not(.banner-content-background) .content-banner, {{WRAPPER}} .promo-banner.banner-content-background .wrapper-content-banner' => 'min-height: {{SIZE}}{{UNIT}};',
				),
				'condition'      => array(
					'style' => 'content-background',
				),
			)
		);

		$this->add_control(
			'increase_spaces',
			array(
				'label'        => esc_html__( 'Increase spaces', 'studiare-plus' ),
				'description'  => esc_html__( 'Suggest to use this option if you have large banners. Padding will be set in percentage to your screen width.', 'studiare-plus' ),
				'type'         => Controls_Manager::SWITCHER,
				'default'      => 'no',
				'label_on'     => esc_html__( 'Yes', 'studiare-plus' ),
				'label_off'    => esc_html__( 'No', 'studiare-plus' ),
				'return_value' => 'yes',
			)
		);

		$this->end_controls_section();
	}

	/**
	 * Render the widget output on the frontend.
	 *
	 * Written in PHP and used to generate the final HTML.
	 *
	 * @since 1.0.0
	 *
	 * @access protected
	 */
	protected function render() {
		woodmart_elementor_banner_template( $this->get_settings_for_display(), $this );
	}
}

Plugin::instance()->widgets_manager->register( new Banner() );
