<?php
namespace Elementor;

if (!defined('ABSPATH')) exit;

class studiare_Widget_Lessons extends Widget_Base {

    public function get_name() {
        return 'lessons';
    }

    public function get_title() {
        return esc_html__('عنصر جلسات دوره', 'studiare-plus');
    }

    public function get_icon() {
        return 'wd-icon-accordion';
    }

    public function get_categories() {
        return ['wd-elements'];
    }

    /* ------------------------------------------------------------
     *  کنترل‌ها
     * ------------------------------------------------------------ */
    protected function register_controls() {

        /* -------------------------------
         *  تنظیمات فصل
         * ------------------------------- */
        $this->start_controls_section(
            'lesson_section',
            [
                'label' => esc_html__('تنظیمات فصل', 'studiare-plus'),
                'type'  => Controls_Manager::SECTION,
            ]
        );

        $this->add_control(
            'image',
            [
                'label'   => __('آیکون فصل (اختیاری)', 'studiare-plus'),
                'type'    => Controls_Manager::MEDIA,
                'default' => ['url' => Utils::get_placeholder_image_src()],
            ]
        );

        $this->add_control(
            'titlelesson',
            [
                'label'   => __('عنوان فصل دوره', 'studiare-plus'),
                'type'    => Controls_Manager::TEXT,
                'default' => 'عنوان فصل نمونه',
            ]
        );

        $this->add_control(
            'subtitlelesson',
            [
                'label'   => __('زیرنویس فصل', 'studiare-plus'),
                'type'    => Controls_Manager::TEXTAREA,
                'default' => 'زیرعنوان نمونه برای این فصل',
            ]
        );

        $this->add_control(
            'arrowsection',
            [
                'label'     => __('قابلیت باز و بسته شدن؟', 'studiare-plus'),
                'type'      => Controls_Manager::SWITCHER,
                'default'   => 'yes',
                'label_on'  => esc_html_x('بله', 'studiare-plus'),
                'label_off' => esc_html_x('خیر', 'studiare-plus'),
            ]
        );

        $this->end_controls_section();


        /* -------------------------------
         *  تنظیمات درس‌ها
         * ------------------------------- */
        $this->start_controls_section(
            'lessons_section',
            [
                'label' => esc_html__('درس‌ها', 'studiare-plus'),
                'type'  => Controls_Manager::SECTION,
            ]
        );

        $repeater = new Repeater();

        /* خصوصی؟ */
        $repeater->add_control(
            'private_lesson',
            [
                'label'   => __('درس خصوصی است؟', 'studiare-plus'),
                'type'    => Controls_Manager::SELECT,
                'default' => 'no',
                'options' => ['yes' => 'بله', 'no' => 'خیر'],
            ]
        );

        /* تاخیر (وابسته به خصوصی بودن) */
        $repeater->add_control(
            'delay_days',
            [
                'label'       => __('قابل مشاهده پس از چند روز؟', 'studiare-plus'),
                'type'        => Controls_Manager::NUMBER,
                'default'     => 0,
                'min'         => 0,
                'condition'   => [
                    'private_lesson' => 'yes'
                ],
                'description' => 'اگر 0 باشد یعنی بلافاصله پس از خرید قابل مشاهده است.',
            ]
        );

        /* عنوان درس */
        $repeater->add_control(
            'subtitlelesson',
            [
                'label'   => __('عنوان درس', 'studiare-plus'),
                'type'    => Controls_Manager::TEXT,
                'default' => 'عنوان درس نمونه',
            ]
        );

        $repeater->add_control(
            'subtitlelesson_sub',
            [
                'label'   => __('زیرنویس درس', 'studiare-plus'),
                'type'    => Controls_Manager::TEXT,
                'default' => 'توضیح کوتاه برای درس',
            ]
        );

        $repeater->add_control(
            'icon',
            [
                'label'   => __('آیکون درس', 'studiare-plus'),
                'type'    => Controls_Manager::ICONS,
                'default' => ['value' => 'fas fa-play', 'library' => 'solid'],
            ]
        );

        $repeater->add_control(
            'label_lesson',
            [
                'label'   => __('لیبل درس', 'studiare-plus'),
                'type'    => Controls_Manager::SELECT,
                'default' => '',
                'options' => [
                    'free'        => 'رایگان',
                    'video'       => 'ویدئو',
                    'exam'        => 'آزمون',
                    'quiz'        => 'کوئیز',
                    'lecture'     => 'مقاله',
                    'practice'    => 'تمرین',
                    'attachments' => 'فایل ضمیمه',
                    'sound'       => 'صوت',
                ],
            ]
        );

        $repeater->add_control(
            'preview_video',
            [
                'label' => __('پیش‌نمایش ویدئو', 'studiare-plus'),
                'type'  => Controls_Manager::URL,
            ]
        );

        $repeater->add_control(
            'download_lesson',
            [
                'label' => __('لینک دانلود فایل', 'studiare-plus'),
                'type'  => Controls_Manager::URL,
            ]
        );

        $repeater->add_control(
            'lesson_content',
            [
                'label' => __('محتوای درس', 'studiare-plus'),
                'type'  => Controls_Manager::WYSIWYG,
            ]
        );

        $this->add_control(
            'lessons_list',
            [
                'label'       => __('لیست دروس', 'studiare-plus'),
                'type'        => Controls_Manager::REPEATER,
                'fields'      => $repeater->get_controls(),
                'title_field' => '{{{subtitlelesson}}}',
            ]
        );

        $this->end_controls_section();
    }


    /* ------------------------------------------------------------
     *  محاسبه روز گذشته از خرید
     * ------------------------------------------------------------ */
    private function get_days_since_purchase($user_id, $product_id) {

        if (!$user_id) return false;

        $orders = wc_get_orders([
            'customer_id' => $user_id,
            'limit'       => -1,
            'return'      => 'objects',
            'status'      => ['completed', 'processing'],
        ]);

        foreach ($orders as $order) {
            foreach ($order->get_items() as $item) {
                if ($item->get_product_id() == $product_id) {
                    $order_time = $order->get_date_created()->getTimestamp();
                    return floor((time() - $order_time) / DAY_IN_SECONDS);
                }
            }
        }

        return false;
    }


    /* ------------------------------------------------------------
     *  رندر
     * ------------------------------------------------------------ */
    protected function render() {

        $settings     = $this->get_settings_for_display();
        $current_user = wp_get_current_user();
        $product_id   = get_queried_object_id();

        /* آیا کاربر خریدار است؟ */
        $bought_course = (
            $current_user->ID &&
            wc_customer_bought_product($current_user->user_login, $current_user->ID, $product_id)
        );

        $days_passed = $this->get_days_since_purchase($current_user->ID, $product_id);
        ?>

        <div class="elementory-section">
            <div class="course-section">

                <!-- عنوان فصل -->
                <div class="course-section-title-elementory<?php echo ($settings['arrowsection'] === 'yes' ? ' cursor-pointer' : ''); ?>">

                    <?php if (!empty($settings['image']['url'])): ?>
                        <img src="<?php echo esc_url($settings['image']['url']); ?>" alt="">
                    <?php endif; ?>

                    <div class="gheadlinel">
                        <span><?php echo esc_html($settings['titlelesson']); ?></span>
                        <p class="subtitle-lesson"><?php echo esc_html($settings['subtitlelesson']); ?></p>
                    </div>

                    <?php if ($settings['arrowsection'] === 'yes'): ?>
                        <i class="fal fa-chevron-down"></i>
                    <?php endif; ?>
                </div>

                <div class="panel-group<?php echo ($settings['arrowsection'] === 'yes' ? ' deactive' : ''); ?>">

                    <?php foreach ($settings['lessons_list'] as $lesson): ?>

                        <?php
                        $private     = $lesson['private_lesson'];
                        $delay_days  = intval($lesson['delay_days']);
                        $badge       = !empty($lesson['label_lesson']) ? $lesson['label_lesson'] : '';
                        $locked_by_delay = false;

                        if ($private === 'yes' && $bought_course) {
                            if ($delay_days > 0 && (!$days_passed || $days_passed < $delay_days)) {
                                $locked_by_delay = true;
                            }
                        }
                        ?>

                        <div class="course-panel-heading">
                            <div class="panel-heading-left">

                                <div class="course-lesson-icon">
                                    <?php Icons_Manager::render_icon($lesson['icon'], ['aria-hidden' => 'true']); ?>
                                </div>

                                <div class="title">

                                    <h4>
                                        <?php echo esc_html($lesson['subtitlelesson']); ?>

                                        <?php if ($badge): ?>
                                            <span class="badge-item <?php echo esc_attr($badge); ?>">
                                                <?php echo esc_html(add_class_value_in_any_lang($badge)); ?>
                                            </span>
                                        <?php endif; ?>
                                    </h4>

                                    <p class="subtitle"><?php echo esc_html($lesson['subtitlelesson_sub']); ?></p>
                                </div>
                            </div>

                            <div class="panel-heading-right">

                                <?php if (!empty($lesson['preview_video']['url'])): ?>
                                    <a class="video-lesson-preview preview-button" href="<?php echo esc_url($lesson['preview_video']['url']); ?>">
                                        <i class="fal fa-play-circle"></i> پیش‌نمایش
                                    </a>
                                <?php endif; ?>

                                <?php if (!empty($lesson['download_lesson']['url'])): ?>
                                    <?php if ($private === 'no' || ($bought_course && !$locked_by_delay)): ?>
                                        <a class="download-button" href="<?php echo esc_url($lesson['download_lesson']['url']); ?>">
                                            <i class="fal fa-download"></i>
                                        </a>
                                    <?php else: ?>
                                        <div class="download-button gray"><i class="fal fa-download"></i></div>
                                    <?php endif; ?>
                                <?php endif; ?>

                                <?php if ($private === 'yes'): ?>
                                    <div class="private-lesson">
                                        <?php if (!$bought_course): ?>
                                            <i class="fal fa-lock"></i> <span>خصوصی</span>
                                        <?php elseif ($locked_by_delay): ?>
                                            <i class="fal fa-clock"></i> 
                                            <span>
                                                فعال پس از <?php echo $delay_days; ?> روز
                                            </span>
                                        <?php else: ?>
                                            <i class="fal fa-unlock green-lock"></i> <span>در دسترس</span>
                                        <?php endif; ?>
                                    </div>
                                <?php endif; ?>

                            </div>
                        </div>

                        <!-- محتوا -->
                        <div class="panel-content">
                            <div class="panel-content-inner">

                                <?php
                                if ($private === 'yes') {

                                    if (!$bought_course) {
                                        echo 'این درس خصوصی است. برای مشاهده باید دوره را خریداری کنید.';
                                    }
                                    elseif ($locked_by_delay) {
                                        echo 'این درس پس از ' . $delay_days . ' روز از ثبت‌نام قابل مشاهده می‌شود.';
                                    }
                                    else {
                                        echo wp_kses_post($lesson['lesson_content']);
                                    }

                                } else {
                                    echo wp_kses_post($lesson['lesson_content']);
                                }
                                ?>

                            </div>
                        </div>

                    <?php endforeach; ?>

                </div>
            </div>
        </div>

        <?php
    }
}

Plugin::instance()->widgets_manager->register(new studiare_Widget_Lessons());
